<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InvoiceBox;
use App\Models\InvoiceBom;
use App\Models\ProductPendukungBox;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class InvoiceBoxController extends Controller
{
    /**
     * Display a listing of invoice boxes
     */
    public function index()
    {
        $invoiceBoxes = InvoiceBox::with('invoiceBom')
            ->latest()
            ->paginate(10);
        
        return view('invoice.box.index', compact('invoiceBoxes'));
    }

    /**
     * Show the form for creating a new invoice box
     */
    public function create()
    {
        // Get all fixed invoice BOMs that don't have box yet
        $invoiceBoms = InvoiceBom::where('is_fixed', true)
            ->whereDoesntHave('invoiceBox')
            ->latest()
            ->get();
        
        // Get product pendukung box for reference (optional)
        $products = ProductPendukungBox::orderBy('material_code')->get();
        
        return view('invoice.box.create', compact('invoiceBoms', 'products'));
    }

    /**
     * Store a newly created invoice box
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_bom_id' => 'required|exists:invoice_boms,id',
            'box_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            // Check if invoice already has box
            $existing = InvoiceBox::where('invoice_bom_id', $request->invoice_bom_id)->first();
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invoice BOM ini sudah memiliki data box!'
                ], 422);
            }

            // Get Invoice BOM dimensions
            $invoiceBom = InvoiceBom::findOrFail($request->invoice_bom_id);
            $dimensions = $invoiceBom->parseSizeDimensions();

            // Create invoice box
            $invoiceBox = InvoiceBox::create([
                'invoice_bom_id' => $request->invoice_bom_id,
                'p' => $dimensions['p'],
                'l' => $dimensions['l'],
                't' => $dimensions['t'],
                'box_data' => $request->box_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Box berhasil dibuat!',
                'invoiceBox' => $invoiceBox
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Box creation error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice box
     */
    public function show($id)
    {
        $invoiceBox = InvoiceBox::with('invoiceBom')->findOrFail($id);
        return view('invoice.box.show', compact('invoiceBox'));
    }

    /**
     * Show the form for editing the specified invoice box
     */
    public function edit($id)
{
    $invoiceBox = InvoiceBox::with('invoiceBom')->findOrFail($id);
    
    // ✅ Ganti nama variable dari $products jadi $komponenPendukungMaster
    $komponenPendukungMaster = ProductPendukungBox::orderBy('material_code')->get();
    
    return view('invoice.box.edit', compact('invoiceBox', 'komponenPendukungMaster'));
}

    /**
     * Update the specified invoice box
     */
    public function update(Request $request, $id)
{
    $invoiceBox = InvoiceBox::findOrFail($id);
    
    // Decode JSON dari request
    $requestData = $request->all();
    
    // Parse box_data jika masih string
    if (isset($requestData['box_data']) && is_string($requestData['box_data'])) {
        $boxData = json_decode($requestData['box_data'], true);
    } else {
        $boxData = $requestData['box_data'] ?? [];
    }
    
    $validator = Validator::make($request->all(), [
        'grand_total' => 'required|numeric|min:0'
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'message' => 'Validation error',
            'errors' => $validator->errors()
        ], 422);
    }

    try {
        DB::beginTransaction();
        
        // Update dimensi jika ada
        $updateData = [
            'box_data' => json_encode($boxData),
            'grand_total' => $request->grand_total
        ];
        
        if ($request->has('p')) {
            $updateData['p'] = $request->p;
        }
        if ($request->has('l')) {
            $updateData['l'] = $request->l;
        }
        if ($request->has('t')) {
            $updateData['t'] = $request->t;
        }
        
        $invoiceBox->update($updateData);

        DB::commit();

        return response()->json([
            'success' => true,
            'message' => 'Invoice Box berhasil diupdate!',
            'invoiceBox' => $invoiceBox
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Invoice Box update error: ' . $e->getMessage());
        Log::error('Stack trace: ' . $e->getTraceAsString());
        
        return response()->json([
            'success' => false,
            'message' => 'Terjadi kesalahan: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Remove the specified invoice box
     */
    public function destroy($id)
    {
        $invoiceBox = InvoiceBox::findOrFail($id);
        $invoiceBox->delete();

        return redirect()->route('invoice.box.index')
            ->with('success', 'Invoice Box berhasil dihapus!');
    }

    /**
 * ✅ CALCULATE SKID BOX REGULER PJG
 */
public static function calculateSkidBoxRegulerPjg($pjgBoxReguler, $qty = 2)
{
    $pjg = $pjgBoxReguler - 10;
    $lbr = 120;
    $tbl = 12;
    $m2 = ($pjg * $lbr) / 1000000;
    $harga = 125000; // Rp 125.000 per M²
    $total = $qty * $harga * $m2;
    
    return [
        'nama' => 'SKID PLYWOOD POPLAR PJG',
        'dimensi' => [
            'pjg' => $pjg,
            'lbr' => $lbr,
            'tbl' => $tbl
        ],
        'm2' => $m2,
        'qty' => $qty,
        'harga' => $harga,
        'total' => $total
    ];
}

/**
 * ✅ CALCULATE SKID BOX REGULER LBR
 */
public static function calculateSkidBoxRegulerLbr($lbrBoxReguler, $qty = 6)
{
    $pjg = $lbrBoxReguler - 10;
    $lbr = 120;
    $tbl = 12;
    $m2 = ($pjg * $lbr) / 1000000;
    $harga = 125000; // Rp 125.000 per M²
    $total = $qty * $harga * $m2;
    
    return [
        'nama' => 'SKID PLYWOOD POPLAR LBR',
        'dimensi' => [
            'pjg' => $pjg,
            'lbr' => $lbr,
            'tbl' => $tbl
        ],
        'm2' => $m2,
        'qty' => $qty,
        'harga' => $harga,
        'total' => $total
    ];
}

/**
 * ✅ CALCULATE SKID BOX TUMBU PJG
 */
public static function calculateSkidBoxTumbuPjg($pjgBoxTumbu, $tgiBoxTumbu, $qty = 2)
{
    $pjg = $pjgBoxTumbu - 10;
    $lbr = $tgiBoxTumbu - 10;
    $tbl = 2.4;
    $m2 = ($pjg * $lbr) / 1000000;
    $harga = 75000; // Rp 75.000 per M²
    $total = $qty * $harga * $m2;
    
    return [
        'nama' => 'SKID PLYWOOD POPLAR PJG',
        'dimensi' => [
            'pjg' => $pjg,
            'lbr' => $lbr,
            'tbl' => $tbl
        ],
        'm2' => $m2,
        'qty' => $qty,
        'harga' => $harga,
        'total' => $total
    ];
}

/**
 * ✅ CALCULATE SKID BOX TUMBU LBR
 */
public static function calculateSkidBoxTumbuLbr($lbrBoxTumbu, $tgiBoxTumbu, $qty = 2)
{
    $pjg = $lbrBoxTumbu - 10;
    $lbr = $tgiBoxTumbu - 10;
    $tbl = 2.4;
    $m2 = ($pjg * $lbr) / 1000000;
    $harga = 75000; // Rp 75.000 per M²
    $total = $qty * $harga * $m2;
    
    return [
        'nama' => 'SKID PLYWOOD POPLAR LBR',
        'dimensi' => [
            'pjg' => $pjg,
            'lbr' => $lbr,
            'tbl' => $tbl
        ],
        'm2' => $m2,
        'qty' => $qty,
        'harga' => $harga,
        'total' => $total
    ];
}

    /**
     * Print Invoice Box
     */
    public function print($id)
    {
        $invoiceBox = InvoiceBox::with('invoiceBom')->findOrFail($id);
        
        return view('invoice.box.print', compact('invoiceBox'));
    }

    /**
     * ✅ CALCULATE BOX REGULER
     * Dimensi:
     * - PJG = P + 122
     * - LBR = L + 122
     * - TGI = T + 122 + 24
     * Ukuran Karton:
     * - PJG = (PJG_dimensi + LBR_dimensi + 59) x 2
     * - LBR = (LBR_dimensi + TGI_dimensi) + 22
     */
    public static function calculateBoxReguler($p, $l, $t, $priceM2, $kebutuhanUnit)
    {
        // Dimensi Box
        $pjg_dimensi = $p + 122;
        $lbr_dimensi = $l + 122;
        $tgi_dimensi = $t + 122 + 24;
        
        // Ukuran Karton
        $pjg_karton = ($pjg_dimensi + $lbr_dimensi + 59) * 2;
        $lbr_karton = ($lbr_dimensi + $tgi_dimensi) + 22;
        $m2 = ($pjg_karton * $lbr_karton) / 1000000;
        
        // Harga
        $priceKarton = $priceM2 * $m2;
        $harga = $kebutuhanUnit * $priceKarton;
        
        return [
            'jenis_box' => 'BOX REGULER',
            'dimensi_box' => [
                'pjg' => $pjg_dimensi,
                'lbr' => $lbr_dimensi,
                'tgi' => $tgi_dimensi
            ],
            'ukuran_karton' => [
                'pjg' => $pjg_karton,
                'lbr' => $lbr_karton,
                'm2' => $m2
            ],
            'price_m2' => $priceM2,
            'price_karton' => $priceKarton,
            'kebutuhan_unit' => $kebutuhanUnit,
            'harga' => $harga
        ];
    }

    /**
     * ✅ CALCULATE BOX TUMBU
     * Dimensi:
     * - PJG = P + 90
     * - LBR = L + 90
     * - TGI = T + 90
     * Ukuran Karton:
     * - PJG = PJG_dimensi + (2 x TGI_dimensi) + 18
     * - LBR = LBR_dimensi + (2 x TGI_dimensi) + 18
     */
    public static function calculateBoxTumbu($p, $l, $t, $priceM2, $kebutuhanUnit)
    {
        // Dimensi Box
        $pjg_dimensi = $p + 90;
        $lbr_dimensi = $l + 90;
        $tgi_dimensi = $t + 90;
        
        // Ukuran Karton
        $pjg_karton = $pjg_dimensi + (2 * $tgi_dimensi) + 18;
        $lbr_karton = $lbr_dimensi + (2 * $tgi_dimensi) + 18;
        $m2 = ($pjg_karton * $lbr_karton) / 1000000;
        
        // Harga
        $priceKarton = $priceM2 * $m2;
        $harga = $kebutuhanUnit * $priceKarton;
        
        return [
            'jenis_box' => 'BOX TUMBU',
            'dimensi_box' => [
                'pjg' => $pjg_dimensi,
                'lbr' => $lbr_dimensi,
                'tgi' => $tgi_dimensi
            ],
            'ukuran_karton' => [
                'pjg' => $pjg_karton,
                'lbr' => $lbr_karton,
                'm2' => $m2
            ],
            'price_m2' => $priceM2,
            'price_karton' => $priceKarton,
            'kebutuhan_unit' => $kebutuhanUnit,
            'harga' => $harga
        ];
    }

    /**
     * ✅ CALCULATE BOX TUTUP
     * Dimensi:
     * - PJG = P + 20
     * - LBR = L + 20
     * - TGI = T + 20
     * Ukuran Karton:
     * - PJG = PJG_dimensi + (2 x TGI_dimensi) + 18
     * - LBR = LBR_dimensi + (2 x TGI_dimensi) + 18
     */
    public static function calculateBoxTutup($p, $l, $t, $priceM2, $kebutuhanUnit)
    {
        // Dimensi Box
        $pjg_dimensi = $p + 20;
        $lbr_dimensi = $l + 20;
        $tgi_dimensi = $t + 20;
        
        // Ukuran Karton
        $pjg_karton = $pjg_dimensi + (2 * $tgi_dimensi) + 18;
        $lbr_karton = $lbr_dimensi + (2 * $tgi_dimensi) + 18;
        $m2 = ($pjg_karton * $lbr_karton) / 1000000;
        
        // Harga
        $priceKarton = $priceM2 * $m2;
        $harga = $kebutuhanUnit * $priceKarton;
        
        return [
            'jenis_box' => 'BOX TUTUP',
            'dimensi_box' => [
                'pjg' => $pjg_dimensi,
                'lbr' => $lbr_dimensi,
                'tgi' => $tgi_dimensi
            ],
            'ukuran_karton' => [
                'pjg' => $pjg_karton,
                'lbr' => $lbr_karton,
                'm2' => $m2
            ],
            'price_m2' => $priceM2,
            'price_karton' => $priceKarton,
            'kebutuhan_unit' => $kebutuhanUnit,
            'harga' => $harga
        ];
    }

    /**
     * ✅ CALCULATE FILLER BOX (Manual Input)
     * Ukuran Karton:
     * - PJG = (PJG_manual + LBR_manual + 59) x 2
     * - LBR = (LBR_manual + TGI_manual) + 22
     */
    public static function calculateFillerBox($pjgManual, $lbrManual, $tgiManual, $priceM2, $kebutuhanUnit)
    {
        // Ukuran Karton
        $pjg_karton = ($pjgManual + $lbrManual + 59) * 2;
        $lbr_karton = ($lbrManual + $tgiManual) + 22;
        $m2 = ($pjg_karton * $lbr_karton) / 1000000;
        
        // Harga
        $priceKarton = $priceM2 * $m2;
        $harga = $kebutuhanUnit * $priceKarton;
        
        return [
            'jenis_box' => 'FILLER BOX',
            'dimensi_box' => [
                'pjg' => $pjgManual,
                'lbr' => $lbrManual,
                'tgi' => $tgiManual
            ],
            'ukuran_karton' => [
                'pjg' => $pjg_karton,
                'lbr' => $lbr_karton,
                'm2' => $m2
            ],
            'price_m2' => $priceM2,
            'price_karton' => $priceKarton,
            'kebutuhan_unit' => $kebutuhanUnit,
            'harga' => $harga
        ];
    }

    /**
     * ✅ CALCULATE LAYER HONEYCOMB
     * Dimensi:
     * - PJG = PJG_FillerBox - 10
     * - LBR = TGI_FillerBox
     * - TGI = Manual Input
     * Ukuran Karton:
     * - PJG = PJG_dimensi
     * - LBR = LBR_dimensi + 21
     */
    public static function calculateLayerHoneycomb($pjgFillerBox, $tgiFillerBox, $tgiManual, $priceM2, $kebutuhanUnit)
    {
        // Dimensi Box
        $pjg_dimensi = $pjgFillerBox - 10;
        $lbr_dimensi = $tgiFillerBox;
        $tgi_dimensi = $tgiManual;
        
        // Ukuran Karton
        $pjg_karton = $pjg_dimensi;
        $lbr_karton = $lbr_dimensi + 21;
        $m2 = ($pjg_karton * $lbr_karton) / 1000000;
        
        // Harga
        $priceKarton = $priceM2 * $m2;
        $harga = $kebutuhanUnit * $priceKarton;
        
        return [
            'jenis_box' => 'LAYER HONEYCOMB',
            'dimensi_box' => [
                'pjg' => $pjg_dimensi,
                'lbr' => $lbr_dimensi,
                'tgi' => $tgi_dimensi
            ],
            'ukuran_karton' => [
                'pjg' => $pjg_karton,
                'lbr' => $lbr_karton,
                'm2' => $m2
            ],
            'price_m2' => $priceM2,
            'price_karton' => $priceKarton,
            'kebutuhan_unit' => $kebutuhanUnit,
            'harga' => $harga
        ];
    }

    /**
     * ✅ CALCULATE FILLER SEGITIGA (Manual Input)
     * Ukuran Karton:
     * - PJG = PJG_manual
     * - LBR = LBR_manual + 21
     */
    public static function calculateFillerSegitiga($pjgManual, $lbrManual, $tgiManual, $priceM2, $kebutuhanUnit)
    {
        // Ukuran Karton
        $pjg_karton = $pjgManual;
        $lbr_karton = $lbrManual + 21;
        $m2 = ($pjg_karton * $lbr_karton) / 1000000;
        
        // Harga
        $priceKarton = $priceM2 * $m2;
        $harga = $kebutuhanUnit * $priceKarton;
        
        return [
            'jenis_box' => 'FILLER SEGITIGA',
            'dimensi_box' => [
                'pjg' => $pjgManual,
                'lbr' => $lbrManual,
                'tgi' => $tgiManual
            ],
            'ukuran_karton' => [
                'pjg' => $pjg_karton,
                'lbr' => $lbr_karton,
                'm2' => $m2
            ],
            'price_m2' => $priceM2,
            'price_karton' => $priceKarton,
            'kebutuhan_unit' => $kebutuhanUnit,
            'harga' => $harga
        ];
    }

    /**
     * ✅ CALCULATE SUPPORT BOX
     * Dimensi:
     * - PJG = PJG_BoxReguler - 10
     * - LBR = LBR_BoxReguler - 10
     * - TGI = Manual Input
     * Ukuran Karton:
     * - PJG = ((2 x PJG_dimensi) + (2 x LBR_BoxReguler)) + 15
     * - LBR = (LBR_dimensi + (2 x TGI_dimensi)) + 15
     */
    public static function calculateSupportBox($pjgBoxReguler, $lbrBoxReguler, $tgiManual, $priceM2, $kebutuhanUnit)
    {
        // Dimensi Box
        $pjg_dimensi = $pjgBoxReguler - 10;
        $lbr_dimensi = $lbrBoxReguler - 10;
        $tgi_dimensi = $tgiManual;
        
        // Ukuran Karton
        $pjg_karton = ((2 * $pjg_dimensi) + (2 * $lbrBoxReguler)) + 15;
        $lbr_karton = ($lbr_dimensi + (2 * $tgi_dimensi)) + 15;
        $m2 = ($pjg_karton * $lbr_karton) / 1000000;
        
        // Harga
        $priceKarton = $priceM2 * $m2;
        $harga = $kebutuhanUnit * $priceKarton;
        
        return [
            'jenis_box' => 'SUPPORT BOX',
            'dimensi_box' => [
                'pjg' => $pjg_dimensi,
                'lbr' => $lbr_dimensi,
                'tgi' => $tgi_dimensi
            ],
            'ukuran_karton' => [
                'pjg' => $pjg_karton,
                'lbr' => $lbr_karton,
                'm2' => $m2
            ],
            'price_m2' => $priceM2,
            'price_karton' => $priceKarton,
            'kebutuhan_unit' => $kebutuhanUnit,
            'harga' => $harga
        ];
    }

    /**
     * ✅ CALCULATE TOP PAD (Manual Input)
     * Ukuran Karton:
     * - PJG = (2 x TGI_manual) + PJG_manual + 10
     * - LBR = LBR_manual + 21
     */
    public static function calculateTopPad($pjgManual, $lbrManual, $tgiManual, $priceM2, $kebutuhanUnit)
    {
        // Ukuran Karton
        $pjg_karton = (2 * $tgiManual) + $pjgManual + 10;
        $lbr_karton = $lbrManual + 21;
        $m2 = ($pjg_karton * $lbr_karton) / 1000000;
        
        // Harga
        $priceKarton = $priceM2 * $m2;
        $harga = $kebutuhanUnit * $priceKarton;
        
        return [
            'jenis_box' => 'TOP PAD',
            'dimensi_box' => [
                'pjg' => $pjgManual,
                'lbr' => $lbrManual,
                'tgi' => $tgiManual
            ],
            'ukuran_karton' => [
                'pjg' => $pjg_karton,
                'lbr' => $lbr_karton,
                'm2' => $m2
            ],
            'price_m2' => $priceM2,
            'price_karton' => $priceKarton,
            'kebutuhan_unit' => $kebutuhanUnit,
            'harga' => $harga
        ];
    }
}