<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InvoiceMetal;
use App\Models\InvoiceBom;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class InvoiceMetalController extends Controller
{
    /**
     * Display a listing of invoice metals
     */
    public function index()
    {
        $invoiceMetals = InvoiceMetal::with('invoiceBom')
            ->latest()
            ->paginate(10);
        
        return view('invoice.metal.index', compact('invoiceMetals'));
    }

    /**
     * Show the form for creating a new invoice metal
     */
    public function create()
    {
        // Get all fixed invoice BOMs that don't have metal yet
        $invoiceBoms = InvoiceBom::where('is_fixed', true)
            ->whereDoesntHave('invoiceMetal')
            ->latest()
            ->get();
        
        return view('invoice.metal.create', compact('invoiceBoms'));
    }

    /**
     * Store a newly created invoice metal
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_bom_id' => 'required|exists:invoice_boms,id',
            'metal_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            // Check if invoice already has metal
            $existing = InvoiceMetal::where('invoice_bom_id', $request->invoice_bom_id)->first();
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invoice BOM ini sudah memiliki data metal!'
                ], 422);
            }

            // Create invoice metal
            $invoiceMetal = InvoiceMetal::create([
                'invoice_bom_id' => $request->invoice_bom_id,
                'metal_data' => $request->metal_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Metal berhasil dibuat!',
                'invoiceMetal' => $invoiceMetal
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Metal creation error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice metal
     */
    public function show($id)
    {
        $invoiceMetal = InvoiceMetal::with('invoiceBom')->findOrFail($id);
        return view('invoice.metal.show', compact('invoiceMetal'));
    }

    /**
     * Show the form for editing the specified invoice metal
     */
    public function edit($id)
    {
        $invoiceMetal = InvoiceMetal::with('invoiceBom')->findOrFail($id);
        
        return view('invoice.metal.edit', compact('invoiceMetal'));
    }

    /**
     * Update the specified invoice metal
     */
    public function update(Request $request, $id)
    {
        $invoiceMetal = InvoiceMetal::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'metal_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            $invoiceMetal->update([
                'metal_data' => $request->metal_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Metal berhasil diupdate!',
                'invoiceMetal' => $invoiceMetal
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Metal update error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified invoice metal
     */
    public function destroy($id)
    {
        $invoiceMetal = InvoiceMetal::findOrFail($id);
        $invoiceMetal->delete();

        return redirect()->route('invoice.metal.index')
            ->with('success', 'Invoice Metal berhasil dihapus!');
    }
}