<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InvoiceUkir;
use App\Models\InvoiceBom;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class InvoiceUkirController extends Controller
{
    /**
     * Display a listing of invoice ukirs
     */
    public function index()
    {
        $invoiceUkirs = InvoiceUkir::with('invoiceBom')
            ->latest()
            ->paginate(10);
        
        return view('invoice.ukir.index', compact('invoiceUkirs'));
    }

    /**
     * Show the form for creating a new invoice ukir
     */
    public function create()
    {
        // Get all fixed invoice BOMs that don't have ukir yet
        $invoiceBoms = InvoiceBom::where('is_fixed', true)
            ->whereDoesntHave('invoiceUkir')
            ->latest()
            ->get();
        
        return view('invoice.ukir.create', compact('invoiceBoms'));
    }

    /**
     * Store a newly created invoice ukir
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_bom_id' => 'required|exists:invoice_boms,id',
            'ukir_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            // Check if invoice already has ukir
            $existing = InvoiceUkir::where('invoice_bom_id', $request->invoice_bom_id)->first();
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invoice BOM ini sudah memiliki data ukir/bubut!'
                ], 422);
            }

            // Create invoice ukir
            $invoiceUkir = InvoiceUkir::create([
                'invoice_bom_id' => $request->invoice_bom_id,
                'ukir_data' => $request->ukir_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Ukir/Bubut berhasil dibuat!',
                'invoiceUkir' => $invoiceUkir
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Ukir creation error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice ukir
     */
    public function show($id)
    {
        $invoiceUkir = InvoiceUkir::with('invoiceBom')->findOrFail($id);
        return view('invoice.ukir.show', compact('invoiceUkir'));
    }

    /**
     * Show the form for editing the specified invoice ukir
     */
    public function edit($id)
    {
        $invoiceUkir = InvoiceUkir::with('invoiceBom')->findOrFail($id);
        
        return view('invoice.ukir.edit', compact('invoiceUkir'));
    }

    /**
     * Update the specified invoice ukir
     */
    public function update(Request $request, $id)
    {
        $invoiceUkir = InvoiceUkir::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'ukir_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            $invoiceUkir->update([
                'ukir_data' => $request->ukir_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Ukir/Bubut berhasil diupdate!',
                'invoiceUkir' => $invoiceUkir
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Ukir update error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified invoice ukir
     */
    public function destroy($id)
    {
        $invoiceUkir = InvoiceUkir::findOrFail($id);
        $invoiceUkir->delete();

        return redirect()->route('invoice.ukir.index')
            ->with('success', 'Invoice Ukir/Bubut berhasil dihapus!');
    }
}