<?php

namespace App\Http\Controllers\Product;

use App\Http\Controllers\Controller;
use App\Models\ProductPendukungBox;
use Illuminate\Http\Request;

class ProductPendukungBoxController extends Controller
{
    public function index()
    {
        $products = ProductPendukungBox::orderBy('created_at', 'desc')->get();
        return view('product.pendukung-box.index', compact('products'));
    }

    public function create()
    {
        return view('product.pendukung-box.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'material_code' => 'required|string|unique:product_pendukung_boxes,material_code',
            'description' => 'required|string',
            'pricing_type' => 'required|in:pcs,m2',
            'harga_asli' => 'required|numeric|min:0',
            'lebar' => 'nullable|numeric|min:0',
            'panjang' => 'nullable|numeric|min:0',
        ]);

        $product = new ProductPendukungBox();
        $product->material_code = $validated['material_code'];
        $product->description = $validated['description'];
        $product->pricing_type = $validated['pricing_type'];
        $product->harga_asli = $validated['harga_asli'];
        $product->lebar = $validated['lebar'];
        $product->panjang = $validated['panjang'];

        // Calculate M2 if Lebar and Panjang provided
        if ($product->lebar && $product->panjang) {
            $product->m2 = $product->calculateM2();
        }

        // Calculate based on pricing type
        if ($product->pricing_type === 'm2') {
            // M2 pricing - must have lebar and panjang
            if (!$product->lebar || !$product->panjang) {
                return back()->withErrors(['pricing_type' => 'Untuk tipe M2, Lebar dan Panjang harus diisi!'])->withInput();
            }

            $product->harga_m2 = $product->calculateHargaM2();
            $product->harga_finished = $product->calculateHargaFinishedM2();
        } else {
            // PCS pricing
            $product->harga_m2 = null;
            $product->harga_finished = $product->calculateHargaFinishedPCS();
        }

        $product->save();

        return redirect()->route('product.pendukung-box.index')->with('success', 'Product Pendukung Box berhasil ditambahkan!');
    }

    public function edit($id)
    {
        $product = ProductPendukungBox::findOrFail($id);
        return view('product.pendukung-box.edit', compact('product'));
    }

    public function update(Request $request, $id)
    {
        $product = ProductPendukungBox::findOrFail($id);

        $validated = $request->validate([
            'material_code' => 'required|string|unique:product_pendukung_boxes,material_code,' . $id,
            'description' => 'required|string',
            'pricing_type' => 'required|in:pcs,m2',
            'harga_asli' => 'required|numeric|min:0',
            'lebar' => 'nullable|numeric|min:0',
            'panjang' => 'nullable|numeric|min:0',
        ]);

        $product->material_code = $validated['material_code'];
        $product->description = $validated['description'];
        $product->pricing_type = $validated['pricing_type'];
        $product->harga_asli = $validated['harga_asli'];
        $product->lebar = $validated['lebar'];
        $product->panjang = $validated['panjang'];

        // Calculate M2 if Lebar and Panjang provided
        if ($product->lebar && $product->panjang) {
            $product->m2 = $product->calculateM2();
        } else {
            $product->m2 = null;
        }

        // Calculate based on pricing type
        if ($product->pricing_type === 'm2') {
            // M2 pricing - must have lebar and panjang
            if (!$product->lebar || !$product->panjang) {
                return back()->withErrors(['pricing_type' => 'Untuk tipe M2, Lebar dan Panjang harus diisi!'])->withInput();
            }

            $product->harga_m2 = $product->calculateHargaM2();
            $product->harga_finished = $product->calculateHargaFinishedM2();
        } else {
            // PCS pricing
            $product->harga_m2 = null;
            $product->harga_finished = $product->calculateHargaFinishedPCS();
        }

        $product->save();

        return redirect()->route('product.pendukung-box.index')->with('success', 'Product Pendukung Box berhasil diupdate!');
    }

    public function destroy($id)
    {
        $product = ProductPendukungBox::findOrFail($id);
        $product->delete();

        return redirect()->route('product.pendukung-box.index')->with('success', 'Product Pendukung Box berhasil dihapus!');
    }
}