<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProductPendukungBox extends Model
{
    use HasFactory;

    protected $fillable = [
        'material_code',
        'description',
        'pricing_type',
        'harga_asli',
        'lebar',
        'panjang',
        'm2',
        'harga_m2',
        'harga_finished'
    ];

    protected $casts = [
        'harga_asli' => 'decimal:2',
        'lebar' => 'decimal:2',
        'panjang' => 'decimal:2',
        'm2' => 'decimal:6',
        'harga_m2' => 'decimal:2',
        'harga_finished' => 'decimal:2'
    ];

    /**
     * Calculate M2 from Lebar x Panjang
     * Formula: (Lebar × Panjang) / 1,000,000
     */
    public function calculateM2()
    {
        if ($this->lebar && $this->panjang) {
            return ($this->lebar * $this->panjang) / 1000000;
        }
        return null;
    }

    /**
     * Calculate Harga/M2 for M2 pricing type
     * Formula: Harga Asli / M2
     */
    public function calculateHargaM2()
    {
        if ($this->pricing_type === 'm2' && $this->m2 && $this->m2 > 0) {
            return $this->harga_asli / $this->m2;
        }
        return null;
    }

    /**
     * Calculate Harga Finished for M2 pricing type
     * Formula: ROUNDUP((Harga M2 / 80%) / 1000, 1) * 1000
     * 
     * Example:
     * Harga M2 = 6,486.5
     * Step 1: 6,486.5 / 0.8 = 8,108.125
     * Step 2: 8,108.125 / 1000 = 8.108125
     * Step 3: ROUNDUP(8.108125, 1) = 8.2
     * Step 4: 8.2 × 1000 = 8,200
     */
    public function calculateHargaFinishedM2()
    {
        if ($this->pricing_type === 'm2' && $this->harga_m2) {
            $step1 = $this->harga_m2 / 0.8;
            $step2 = $step1 / 1000;
            $step3 = ceil($step2 * 10) / 10; // ROUNDUP to 1 decimal
            return $step3 * 1000;
        }
        return null;
    }

    /**
     * Calculate Harga Finished for PCS pricing type
     * Formula: ROUNDUP((Harga Asli / 60%) / 1000, 1) * 1000
     * 
     * Example:
     * Harga Asli = 3,712
     * Step 1: 3,712 / 0.6 = 6,186.67
     * Step 2: 6,186.67 / 1000 = 6.18667
     * Step 3: ROUNDUP(6.18667, 1) = 6.2
     * Step 4: 6.2 × 1000 = 6,200
     */
    public function calculateHargaFinishedPCS()
    {
        if ($this->pricing_type === 'pcs') {
            $step1 = $this->harga_asli / 0.6;
            $step2 = $step1 / 1000;
            $step3 = ceil($step2 * 10) / 10; // ROUNDUP to 1 decimal
            return $step3 * 1000;
        }
        return null;
    }

    /**
     * Auto calculate Harga Finished based on pricing type
     */
    public function autoCalculateHargaFinished()
    {
        if ($this->pricing_type === 'pcs') {
            return $this->calculateHargaFinishedPCS();
        } else {
            return $this->calculateHargaFinishedM2();
        }
    }
}