@extends('layouts.app')

@section('title', 'Edit Invoice Hardware - Cost Furniture')

@section('page-title', 'Edit Invoice Hardware')

@section('content')

<div class="card modern-card">
    <div class="card-header">
        <h5><i class="fas fa-wrench me-2"></i>Form Edit Invoice Hardware #{{ $invoiceHardware->id }}</h5>
    </div>
    <div class="card-body">
        <form id="hardwareForm">
            @csrf
            @method('PUT')
            
            <!-- Step 1: Invoice BOM Info (READ ONLY) -->
            <div class="section-card">
                <h6 class="section-title"><i class="fas fa-file-invoice me-2"></i>INVOICE BOM INFO</h6>
                
                <div class="info-box">
                    <div class="row">
                        <div class="col-md-3">
                            <strong>Product Name:</strong>
                            <div>{{ $invoiceHardware->invoiceBom->product_name }}</div>
                        </div>
                        <div class="col-md-3">
                            <strong>Product Code:</strong>
                            <div>{{ $invoiceHardware->invoiceBom->product_code }}</div>
                        </div>
                        <div class="col-md-3">
                            <strong>Product Type:</strong>
                            <div>{{ $invoiceHardware->invoiceBom->product_type }}</div>
                        </div>
                        <div class="col-md-3">
                            <strong>BOM Grand Total:</strong>
                            <div class="text-success fw-bold">{{ \App\Helpers\CasegoodHelper::formatRupiah($invoiceHardware->invoiceBom->grand_total) }}</div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Step 2: Hardware Groups -->
            <div class="section-card">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h6 class="section-title mb-0"><i class="fas fa-boxes me-2"></i>HARDWARE GROUPS</h6>
                    <button type="button" class="btn btn-sm btn-success" onclick="addHardwareGroup()">
                        <i class="fas fa-plus me-2"></i>Tambah Group
                    </button>
                </div>
                
                <div id="hardware-groups-container">
                    <!-- Existing groups will be loaded here -->
                </div>
                
                <div id="no-groups-message" class="text-center py-4" style="display: none;">
                    <i class="fas fa-info-circle fa-2x text-muted mb-2"></i>
                    <p class="text-muted">Belum ada hardware group. Klik "Tambah Group" untuk mulai.</p>
                </div>
            </div>
            
            <!-- ✅ NEW: Tambahan Manual (Standalone - Tanpa Group) -->
            <div class="section-card">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h6 class="section-title mb-0">
                        <i class="fas fa-edit me-2"></i>TAMBAHAN MANUAL (Tanpa Group)
                    </h6>
                    <button type="button" class="btn btn-sm btn-warning" onclick="addManualItemStandalone()">
                        <i class="fas fa-plus-circle me-2"></i>Tambah Item Manual
                    </button>
                </div>
                
                <div id="manual-items-standalone-container">
                    <!-- Manual items will be loaded here -->
                </div>
                
                <div id="no-manual-items-message" class="text-center py-4">
                    <i class="fas fa-info-circle fa-2x text-muted mb-2"></i>
                    <p class="text-muted">Belum ada item manual. Klik "Tambah Item Manual" untuk mulai.</p>
                </div>
                
                <!-- Subtotal Manual Items -->
                <div class="subtotal-box mt-3" id="manual-subtotal-box" style="display: none;">
                    <div class="row align-items-center">
                        <div class="col-md-6">
                            <strong><i class="fas fa-calculator me-2"></i>Subtotal Tambahan Manual</strong>
                        </div>
                        <div class="col-md-6 text-end">
                            <h5 class="mb-0 text-primary" id="manual-items-subtotal">Rp 0</h5>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Grand Total -->
            <div class="section-card bg-gradient">
                <div class="row align-items-center">
                    <div class="col-md-6">
                        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>GRAND TOTAL HARDWARE</h5>
                    </div>
                    <div class="col-md-6 text-end">
                        <h3 class="mb-0 text-white" id="grand-total-display">Rp 0</h3>
                        <input type="hidden" name="grand_total" id="grand_total" value="0">
                    </div>
                </div>
            </div>
            
            <!-- Action Buttons -->
            <div class="d-flex gap-2 justify-content-end">
                <a href="{{ route('invoice.hardware.index') }}" class="btn btn-secondary">
                    <i class="fas fa-times me-2"></i>Batal
                </a>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save me-2"></i>Update Invoice Hardware
                </button>
            </div>
        </form>
    </div>
</div>

<!-- ✅ Modal Input Manual Material (Untuk Group) -->
<div class="modal fade" id="manualMaterialModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-warning text-white">
                <h5 class="modal-title"><i class="fas fa-edit me-2"></i>Tambah Material Manual</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" id="manual-group-id">
                
                <div class="mb-3">
                    <label class="form-label">Material Code <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="manual-material-code" placeholder="e.g. CUSTOM-001" required>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Material Description <span class="text-danger">*</span></label>
                    <textarea class="form-control" id="manual-material-desc" rows="2" placeholder="Deskripsi material..." required></textarea>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="manual-material-qty" min="1" value="1" required>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Price <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="manual-material-price" min="0" step="0.01" placeholder="0" required>
                    </div>
                </div>
                
                <div class="mt-3">
                    <strong>Total:</strong> <span id="manual-material-total" class="text-success fw-bold">Rp 0</span>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-warning" onclick="saveManualMaterialToGroup()">
                    <i class="fas fa-save me-2"></i>Simpan
                </button>
            </div>
        </div>
    </div>
</div>

<!-- ✅ Modal Input Manual Item (Standalone) -->
<div class="modal fade" id="manualItemModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-warning text-white">
                <h5 class="modal-title"><i class="fas fa-plus-circle me-2"></i>Tambah Item Manual (Tanpa Group)</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <input type="hidden" id="standalone-edit-index">
                
                <div class="mb-3">
                    <label class="form-label">Material Code <span class="text-danger">*</span></label>
                    <input type="text" class="form-control" id="standalone-material-code" placeholder="e.g. MANUAL-001" required>
                </div>
                
                <div class="mb-3">
                    <label class="form-label">Material Description <span class="text-danger">*</span></label>
                    <textarea class="form-control" id="standalone-material-desc" rows="2" placeholder="Deskripsi material..." required></textarea>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="standalone-material-qty" min="1" value="1" required>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Price <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="standalone-material-price" min="0" step="0.01" placeholder="0" required>
                    </div>
                </div>
                
                <div class="mt-3">
                    <strong>Total:</strong> <span id="standalone-material-total" class="text-success fw-bold">Rp 0</span>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-warning" onclick="saveManualItemStandalone()">
                    <i class="fas fa-save me-2"></i>Simpan
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay" style="display: none;">
    <div class="spinner-border text-light" role="status">
        <span class="visually-hidden">Loading...</span>
    </div>
    <p class="text-light mt-3">Mengupdate invoice hardware...</p>
</div>

<style>
/* Modern Card */
.modern-card {
    border: none;
    border-radius: 15px;
    box-shadow: 0 5px 20px rgba(0,0,0,0.05);
}

.modern-card .card-header {
    background: linear-gradient(135deg, #f39c12 0%, #e67e22 100%);
    color: white;
    border-radius: 15px 15px 0 0;
    padding: 20px 25px;
    border: none;
}

.modern-card .card-header h5 {
    margin: 0;
    font-weight: 700;
    color: white;
}

.modern-card .card-body {
    padding: 30px;
}

/* Section Card */
.section-card {
    background: #fff;
    border: 2px solid #f0f0f0;
    border-radius: 12px;
    padding: 25px;
    margin-bottom: 25px;
}

.section-title {
    font-weight: 700;
    color: #f39c12;
    font-size: 16px;
    margin-bottom: 20px;
}

/* Info Box */
.info-box {
    background: linear-gradient(135deg, rgba(243, 156, 18, 0.1) 0%, rgba(230, 126, 34, 0.1) 100%);
    border: 2px solid rgba(243, 156, 18, 0.2);
    border-radius: 10px;
    padding: 20px;
}

.info-box strong {
    color: #666;
    font-size: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.info-box div:not(strong) {
    color: #2c3e50;
    font-size: 16px;
    font-weight: 600;
    margin-top: 5px;
}

/* Hardware Group Card */
.hardware-group-card {
    background: #fff;
    border: 2px solid #e9ecef;
    border-left: 5px solid #f39c12;
    border-radius: 10px;
    padding: 20px;
    margin-bottom: 20px;
    transition: all 0.3s ease;
}

.hardware-group-card:hover {
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
    transform: translateY(-2px);
}

.group-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding-bottom: 15px;
    border-bottom: 2px solid #f0f0f0;
}

.group-title {
    font-size: 16px;
    font-weight: 700;
    color: #2c3e50;
}

/* Search Section */
.search-section {
    background: #f8f9fa;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    padding: 15px;
    margin-bottom: 15px;
}

.search-results {
    max-height: 200px;
    overflow-y: auto;
    border: 1px solid #dee2e6;
    border-radius: 5px;
    background: white;
}

.search-results select {
    width: 100%;
    cursor: pointer;
    border: none;
    outline: none;
}

.search-results select option {
    padding: 8px;
    cursor: pointer;
    font-size: 13px;
}

.search-results select option:hover {
    background: #f39c12;
    color: white;
}

/* Selected Materials Section */
.selected-materials-section {
    min-height: 120px;
}

.selected-materials-list {
    min-height: 100px;
    max-height: 400px;
    overflow-y: auto;
}

/* Material Card */
.material-card {
    background: #fff;
    border: 2px solid #e9ecef;
    border-left: 4px solid #f39c12;
    border-radius: 8px;
    padding: 15px;
    margin-bottom: 10px;
    transition: all 0.2s ease;
    position: relative;
}

.material-card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    transform: translateY(-1px);
}

/* ✅ Manual Material Card */
.material-card.manual {
    border-left: 4px solid #f39c12;
    background: linear-gradient(135deg, rgba(243, 156, 18, 0.05) 0%, rgba(230, 126, 34, 0.05) 100%);
}

.material-code {
    font-size: 11px;
    color: #666;
    font-weight: 600;
    background: #f8f9fa;
    padding: 2px 6px;
    border-radius: 3px;
    display: inline-block;
    margin-bottom: 5px;
}

.material-desc {
    font-size: 14px;
    color: #2c3e50;
    font-weight: 500;
    margin-bottom: 5px;
}

.material-price {
    font-size: 12px;
    color: #27ae60;
    font-weight: 600;
}

.material-qty,
.material-total {
    min-width: 120px;
}

.material-total {
    text-align: right;
}

.material-total .fw-bold {
    font-size: 14px;
    color: #f39c12;
}

.material-actions {
    display: flex;
    align-items: center;
}

/* ✅ Manual Badge */
.badge-manual {
    position: absolute;
    top: 10px;
    right: 10px;
    background: #f39c12;
    color: white;
    padding: 3px 8px;
    border-radius: 4px;
    font-size: 10px;
    font-weight: 700;
    text-transform: uppercase;
}

/* Subtotal Box */
.subtotal-box {
    background: linear-gradient(135deg, rgba(243, 156, 18, 0.15) 0%, rgba(230, 126, 34, 0.15) 100%);
    border: 2px solid #f39c12;
    border-radius: 8px;
    padding: 15px 20px;
}

/* Grand Total */
.bg-gradient {
    background: linear-gradient(135deg, #f39c12 0%, #e67e22 100%);
    color: white;
}

/* Form Controls */
.form-label {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
    font-size: 14px;
}

.form-control, .form-select {
    padding: 12px 15px;
    border: 2px solid #e0e0e0;
    border-radius: 8px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-control:focus, .form-select:focus {
    border-color: #f39c12;
    box-shadow: 0 0 0 0.2rem rgba(243, 156, 18, 0.1);
    outline: none;
}

.form-control-sm {
    padding: 6px 10px;
    font-size: 13px;
}

/* Input Group */
.input-group .btn {
    border: 2px solid #e0e0e0;
}

.input-group .btn:hover {
    background: #e74c3c;
    border-color: #e74c3c;
    color: white;
}

/* Buttons */
.btn {
    padding: 12px 24px;
    border-radius: 8px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.btn-primary {
    background: linear-gradient(135deg, #f39c12 0%, #e67e22 100%);
    border: none;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #e67e22 0%, #d35400 100%);
}

.btn-warning {
    background: linear-gradient(135deg, #f39c12 0%, #e67e22 100%);
    border: none;
    color: white;
}

.btn-warning:hover {
    background: linear-gradient(135deg, #e67e22 0%, #d35400 100%);
    color: white;
}

.btn-sm {
    padding: 6px 12px;
    font-size: 13px;
}

/* Loading Overlay */
.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    z-index: 9999;
}

/* Empty State */
.text-muted {
    color: #6c757d !important;
}

/* Modal */
.modal-header.bg-warning {
    background: linear-gradient(135deg, #f39c12 0%, #e67e22 100%) !important;
}
</style>

<script>
// ============================================
// HARDWARE EDIT - COMPLETE FIXED JAVASCRIPT
// ============================================

// ============================================
// GLOBAL VARIABLES (WINDOW SCOPE)
// ============================================

const hardwareData = @json($hardwareData);
const hardwareTypes = @json($hardwareTypes);

// ✅ CRITICAL: Use window scope for persistence
window.existingData = { groups: [], manual_items: [] };
window.hardwareGroups = {};
window.manualItemsStandalone = [];
window.groupCounter = 0;

// ✅ Parse existing data from database
try {
    const rawData = @json($invoiceHardware->hardware_data);
    
    console.log('[INIT] Raw Data Type:', typeof rawData);
    console.log('[INIT] Raw Data:', rawData);
    
    if (typeof rawData === 'string') {
        window.existingData = JSON.parse(rawData);
        console.log('[INIT] ✓ Parsed from string');
    } else if (rawData && typeof rawData === 'object') {
        // Create clean copy
        window.existingData = {
            groups: rawData.groups || [],
            manual_items: rawData.manual_items || []
        };
        console.log('[INIT] ✓ Used object directly');
    }
    
    console.log('[INIT] Existing Data:', window.existingData);
    console.log('[INIT] Groups:', window.existingData.groups?.length || 0);
    console.log('[INIT] Manual Items:', window.existingData.manual_items?.length || 0);
    
} catch (e) {
    console.error('[INIT] ❌ Failed to parse existing data:', e);
    window.existingData = { groups: [], manual_items: [] };
}

// Initialize local variables from window
let hardwareGroups = window.hardwareGroups;
let manualItemsStandalone = window.manualItemsStandalone;
let groupCounter = window.groupCounter;

console.log('[INIT] Hardware Types:', hardwareTypes);
console.log('[INIT] Hardware Data:', hardwareData);

// ============================================
// LOAD EXISTING DATA ON PAGE LOAD
// ============================================

document.addEventListener('DOMContentLoaded', function() {
    console.log('[DOM] === HARDWARE EDIT PAGE LOADED ===');
    console.log('[DOM] Existing Data:', window.existingData);
    
    const existingData = window.existingData;
    
    // ✅ Load existing groups
    if (existingData && existingData.groups && existingData.groups.length > 0) {
        console.log(`[DOM] Loading ${existingData.groups.length} groups...`);
        
        existingData.groups.forEach((group, groupIdx) => {
            console.log(`[DOM] Loading Group ${groupIdx + 1}:`, group);
            
            addHardwareGroup();
            const groupId = `group-${groupCounter}`;
            
            // Set hardware type
            const typeSelect = document.getElementById(`hardware-type-${groupId}`);
            if (typeSelect) {
                typeSelect.value = group.hardware_type;
                loadHardwareMaterials(groupId);
            }
            
            // Load existing materials (both from DB and manual)
            setTimeout(() => {
                const type = group.hardware_type;
                const allMaterials = hardwareData[type] || [];
                
                console.log(`[DOM] Group ${groupId} - Type: ${type}, Available Materials:`, allMaterials.length);
                
                group.materials.forEach((material, matIdx) => {
                    console.log(`[DOM] Loading Material ${matIdx + 1}:`, material);
                    
                    if (material.is_manual) {
                        // ✅ Load manual material
                        console.log('[DOM] → Adding MANUAL material');
                        const materialIndex = `manual-${Date.now()}-${Math.random()}`;
                        addManualMaterialToGroupFromData(groupId, materialIndex, material);
                    } else {
                        // ✅ Load material from database
                        console.log('[DOM] → Adding DATABASE material');
                        const materialIndex = allMaterials.findIndex(m => m.material_code === material.material_code);
                        
                        if (materialIndex !== -1) {
                            const materialObj = allMaterials[materialIndex];
                            addSelectedMaterial(groupId, materialIndex, materialObj);
                            
                            // Set qty after material is added
                            setTimeout(() => {
                                const qtyInput = document.getElementById(`material-qty-${groupId}-${materialIndex}`);
                                if (qtyInput) {
                                    qtyInput.value = material.qty;
                                    updateMaterialQty(groupId, materialIndex);
                                    console.log(`[DOM] ✓ Set qty ${material.qty} for material ${materialIndex}`);
                                }
                            }, 150);
                        } else {
                            console.warn(`[DOM] ⚠️ Material not found in database: ${material.material_code}`);
                        }
                    }
                });
                
                // Update subtotal after all materials loaded
                setTimeout(() => {
                    updateGroupSubtotal(groupId);
                }, 300);
                
            }, 250 * (groupIdx + 1)); // Stagger loading for each group
        });
    } else {
        console.warn('[DOM] ⚠️ No groups found in existing data');
    }
    
    // ✅ Load existing manual standalone items
    setTimeout(() => {
        if (existingData && existingData.manual_items && existingData.manual_items.length > 0) {
            console.log('[DOM] Loading Manual Standalone Items:', existingData.manual_items);
            window.manualItemsStandalone = existingData.manual_items;
            manualItemsStandalone = window.manualItemsStandalone;
            renderManualItemsStandalone();
            updateManualItemsSubtotal();
            console.log('[DOM] ✓ Manual standalone items loaded');
        } else {
            console.log('[DOM] ℹ️ No manual standalone items');
        }
        
        // Update grand total after everything loaded
        setTimeout(() => {
            updateGrandTotal();
            console.log('[DOM] ✓ All data loaded, grand total updated');
        }, 200);
        
    }, 1000); // Wait for all groups to load first
});

// ============================================
// HELPER: Add manual material from existing data
// ============================================

function addManualMaterialToGroupFromData(groupId, materialIndex, material) {
    const selectedContainer = document.getElementById(`selected-materials-${groupId}`);
    
    // Remove "no materials" message if exists
    if (selectedContainer.querySelector('.text-muted')) {
        selectedContainer.innerHTML = '';
    }
    
    const materialCardHtml = `
        <div class="material-card manual" id="material-card-${groupId}-${materialIndex}">
            <span class="badge-manual">MANUAL</span>
            <div class="d-flex align-items-start gap-3">
                <div class="material-info flex-grow-1">
                    <div class="material-code">${material.material_code}</div>
                    <div class="material-desc">${material.material_description}</div>
                    <div class="material-price">@ ${formatRupiah(material.price)}</div>
                </div>
                <div class="material-qty">
                    <label class="form-label small mb-1">Qty:</label>
                    <input type="number" 
                           class="form-control form-control-sm" 
                           id="material-qty-${groupId}-${materialIndex}"
                           placeholder="Qty"
                           min="1"
                           value="${material.qty}"
                           onchange="updateManualMaterialQty('${groupId}', '${materialIndex}')"
                           style="width: 100px;">
                </div>
                <div class="material-total">
                    <label class="form-label small mb-1">Total:</label>
                    <div class="fw-bold" id="material-total-${groupId}-${materialIndex}">${formatRupiah(material.total)}</div>
                </div>
                <div class="material-actions">
                    <button type="button" 
                            class="btn btn-sm btn-danger mt-4" 
                            onclick="removeSelectedMaterial('${groupId}', '${materialIndex}')"
                            title="Hapus">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        </div>
    `;
    
    selectedContainer.insertAdjacentHTML('beforeend', materialCardHtml);
    
    // Add to materials array
    if (!hardwareGroups[groupId]) {
        hardwareGroups[groupId] = { type: '', materials: [], subtotal: 0 };
    }
    
    hardwareGroups[groupId].materials.push({
        index: materialIndex,
        material_code: material.material_code,
        material_description: material.material_description,
        qty: material.qty,
        price: material.price,
        total: material.total,
        is_manual: true
    });
    
    updateGroupSubtotal(groupId);
}

// ============================================
// ADD HARDWARE GROUP
// ============================================

function addHardwareGroup() {
    groupCounter++;
    window.groupCounter = groupCounter;
    const groupId = `group-${groupCounter}`;
    
    // Hide no groups message
    document.getElementById('no-groups-message').style.display = 'none';
    
    const groupHtml = `
        <div class="hardware-group-card" id="${groupId}">
            <div class="group-header">
                <div class="group-title">
                    <i class="fas fa-layer-group me-2"></i>Hardware Group #${groupCounter}
                </div>
                <button type="button" class="btn btn-sm btn-danger" onclick="removeHardwareGroup('${groupId}')">
                    <i class="fas fa-trash me-1"></i>Hapus Group
                </button>
            </div>
            
            <div class="row mb-3">
                <div class="col-md-12">
                    <label class="form-label">Pilih Hardware Type</label>
                    <select class="form-select" id="hardware-type-${groupId}" onchange="loadHardwareMaterials('${groupId}')">
                        <option value="">-- Pilih Hardware Type --</option>
                        ${hardwareTypes.map(type => `<option value="${type}">${type}</option>`).join('')}
                    </select>
                </div>
            </div>
            
            <div id="materials-container-${groupId}" style="display: none;">
                <div id="materials-list-${groupId}">
                    <!-- Search & Selected materials will be loaded here -->
                </div>
            </div>
        </div>
    `;
    
    document.getElementById('hardware-groups-container').insertAdjacentHTML('beforeend', groupHtml);
    
    // Initialize group data
    hardwareGroups[groupId] = {
        type: '',
        materials: [],
        subtotal: 0
    };
    
    window.hardwareGroups = hardwareGroups;
}

// ============================================
// REMOVE HARDWARE GROUP
// ============================================

function removeHardwareGroup(groupId) {
    if (confirm('Yakin ingin menghapus group ini?')) {
        document.getElementById(groupId).remove();
        delete hardwareGroups[groupId];
        window.hardwareGroups = hardwareGroups;
        
        // Check if no groups left
        if (Object.keys(hardwareGroups).length === 0) {
            document.getElementById('no-groups-message').style.display = 'block';
        }
        
        updateGrandTotal();
    }
}

// ============================================
// LOAD HARDWARE MATERIALS (SEARCH MODE)
// ============================================

function loadHardwareMaterials(groupId) {
    const typeSelect = document.getElementById(`hardware-type-${groupId}`);
    const selectedType = typeSelect.value;
    
    if (!selectedType) {
        document.getElementById(`materials-container-${groupId}`).style.display = 'none';
        return;
    }
    
    // Update group type
    hardwareGroups[groupId].type = selectedType;
    window.hardwareGroups = hardwareGroups;
    
    // Keep existing materials if any
    if (!hardwareGroups[groupId].materials) {
        hardwareGroups[groupId].materials = [];
    }
    
    // Show search & selected materials section
    const searchHtml = `
        <div class="search-section">
            <label class="form-label">Cari Material:</label>
            <div class="input-group">
                <input type="text" 
                       class="form-control" 
                       id="search-material-${groupId}"
                       placeholder="Ketik nama material untuk mencari..."
                       onkeyup="searchMaterial('${groupId}')">
                <button class="btn btn-secondary" type="button" onclick="clearSearch('${groupId}')">
                    <i class="fas fa-times"></i> Clear
                </button>
            </div>
            
            <div id="search-results-${groupId}" class="search-results mt-2" style="display: none;">
                <select class="form-select" id="search-dropdown-${groupId}" size="5" onchange="selectMaterialFromDropdown('${groupId}')">
                    <!-- Search results will appear here -->
                </select>
                <small class="text-muted d-block mt-1">Pilih material dari hasil pencarian</small>
            </div>
        </div>
        
        <!-- ✅ Tombol Tambah Material Manual -->
        <div class="mb-3">
            <button type="button" class="btn btn-sm btn-warning" onclick="addManualMaterialToGroup('${groupId}')">
                <i class="fas fa-plus-circle me-2"></i>Tambah Material Manual
            </button>
        </div>
        
        <div class="selected-materials-section mt-3">
            <label class="form-label">Material yang Dipilih:</label>
            <div id="selected-materials-${groupId}" class="selected-materials-list">
                <div class="text-muted text-center p-3" style="background: #f8f9fa; border-radius: 8px;">
                    <i class="fas fa-info-circle me-2"></i>Belum ada material yang dipilih
                </div>
            </div>
        </div>
        
        <div class="subtotal-box">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <strong><i class="fas fa-calculator me-2"></i>Subtotal Group #${groupCounter}</strong>
                </div>
                <div class="col-md-6 text-end">
                    <h5 class="mb-0 text-primary" id="subtotal-${groupId}">Rp 0</h5>
                </div>
            </div>
        </div>
    `;
    
    document.getElementById(`materials-list-${groupId}`).innerHTML = searchHtml;
    document.getElementById(`materials-container-${groupId}`).style.display = 'block';
}

// ============================================
// TAMBAH MATERIAL MANUAL KE GROUP
// ============================================

function addManualMaterialToGroup(groupId) {
    // Reset modal
    document.getElementById('manual-group-id').value = groupId;
    document.getElementById('manual-material-code').value = '';
    document.getElementById('manual-material-desc').value = '';
    document.getElementById('manual-material-qty').value = 1;
    document.getElementById('manual-material-price').value = '';
    document.getElementById('manual-material-total').textContent = 'Rp 0';
    
    // Show modal
    const modal = new bootstrap.Modal(document.getElementById('manualMaterialModal'));
    modal.show();
}

// Auto calculate total in modal
document.getElementById('manual-material-qty').addEventListener('input', calculateManualMaterialTotal);
document.getElementById('manual-material-price').addEventListener('input', calculateManualMaterialTotal);

function calculateManualMaterialTotal() {
    const qty = parseFloat(document.getElementById('manual-material-qty').value) || 0;
    const price = parseFloat(document.getElementById('manual-material-price').value) || 0;
    const total = qty * price;
    
    document.getElementById('manual-material-total').textContent = formatRupiah(total);
}

function saveManualMaterialToGroup() {
    const groupId = document.getElementById('manual-group-id').value;
    const materialCode = document.getElementById('manual-material-code').value.trim();
    const materialDesc = document.getElementById('manual-material-desc').value.trim();
    const qty = parseFloat(document.getElementById('manual-material-qty').value) || 0;
    const price = parseFloat(document.getElementById('manual-material-price').value) || 0;
    
    if (!materialCode || !materialDesc || qty <= 0 || price <= 0) {
        alert('❌ Semua field harus diisi dengan benar!');
        return;
    }
    
    const total = qty * price;
    const materialIndex = `manual-${Date.now()}`;
    
    const selectedContainer = document.getElementById(`selected-materials-${groupId}`);
    
    // Remove "no materials" message if exists
    if (selectedContainer.querySelector('.text-muted')) {
        selectedContainer.innerHTML = '';
    }
    
    // Add manual material card with badge
    const materialCardHtml = `
        <div class="material-card manual" id="material-card-${groupId}-${materialIndex}">
            <span class="badge-manual">MANUAL</span>
            <div class="d-flex align-items-start gap-3">
                <div class="material-info flex-grow-1">
                    <div class="material-code">${materialCode}</div>
                    <div class="material-desc">${materialDesc}</div>
                    <div class="material-price">@ ${formatRupiah(price)}</div>
                </div>
                <div class="material-qty">
                    <label class="form-label small mb-1">Qty:</label>
                    <input type="number" 
                           class="form-control form-control-sm" 
                           id="material-qty-${groupId}-${materialIndex}"
                           placeholder="Qty"
                           min="1"
                           value="${qty}"
                           onchange="updateManualMaterialQty('${groupId}', '${materialIndex}')"
                           style="width: 100px;">
                </div>
                <div class="material-total">
                    <label class="form-label small mb-1">Total:</label>
                    <div class="fw-bold" id="material-total-${groupId}-${materialIndex}">${formatRupiah(total)}</div>
                </div>
                <div class="material-actions">
                    <button type="button" 
                            class="btn btn-sm btn-danger mt-4" 
                            onclick="removeSelectedMaterial('${groupId}', '${materialIndex}')"
                            title="Hapus">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        </div>
    `;
    
    selectedContainer.insertAdjacentHTML('beforeend', materialCardHtml);
    
    // Add to materials array with is_manual flag
    const materialData = {
        index: materialIndex,
        material_code: materialCode,
        material_description: materialDesc,
        qty: qty,
        price: price,
        total: total,
        is_manual: true
    };
    
    hardwareGroups[groupId].materials.push(materialData);
    window.hardwareGroups = hardwareGroups;
    
    updateGroupSubtotal(groupId);
    
    // Close modal
    bootstrap.Modal.getInstance(document.getElementById('manualMaterialModal')).hide();
    
    console.log('✅ Manual material added to group:', materialData);
}

function updateManualMaterialQty(groupId, materialIndex) {
    const qtyInput = document.getElementById(`material-qty-${groupId}-${materialIndex}`);
    if (!qtyInput) return;
    
    const qty = parseInt(qtyInput.value) || 1;
    
    // Find material in array
    const material = hardwareGroups[groupId].materials.find(m => m.index === materialIndex);
    
    if (material) {
        const total = material.price * qty;
        
        document.getElementById(`material-total-${groupId}-${materialIndex}`).textContent = formatRupiah(total);
        
        material.qty = qty;
        material.total = total;
        
        window.hardwareGroups = hardwareGroups;
        updateGroupSubtotal(groupId);
    }
}

// ============================================
// TAMBAH MANUAL ITEM STANDALONE
// ============================================

let standaloneEditIndex = null;

function addManualItemStandalone() {
    standaloneEditIndex = null;
    
    // Reset modal
    document.getElementById('standalone-edit-index').value = '';
    document.getElementById('standalone-material-code').value = '';
    document.getElementById('standalone-material-desc').value = '';
    document.getElementById('standalone-material-qty').value = 1;
    document.getElementById('standalone-material-price').value = '';
    document.getElementById('standalone-material-total').textContent = 'Rp 0';
    
    // Show modal
    const modal = new bootstrap.Modal(document.getElementById('manualItemModal'));
    modal.show();
}

// Auto calculate total in modal
document.getElementById('standalone-material-qty').addEventListener('input', calculateStandaloneMaterialTotal);
document.getElementById('standalone-material-price').addEventListener('input', calculateStandaloneMaterialTotal);

function calculateStandaloneMaterialTotal() {
    const qty = parseFloat(document.getElementById('standalone-material-qty').value) || 0;
    const price = parseFloat(document.getElementById('standalone-material-price').value) || 0;
    const total = qty * price;
    
    document.getElementById('standalone-material-total').textContent = formatRupiah(total);
}

function saveManualItemStandalone() {
    const materialCode = document.getElementById('standalone-material-code').value.trim();
    const materialDesc = document.getElementById('standalone-material-desc').value.trim();
    const qty = parseFloat(document.getElementById('standalone-material-qty').value) || 0;
    const price = parseFloat(document.getElementById('standalone-material-price').value) || 0;
    
    if (!materialCode || !materialDesc || qty <= 0 || price <= 0) {
        alert('❌ Semua field harus diisi dengan benar!');
        return;
    }
    
    const total = qty * price;
    
    const editIndex = document.getElementById('standalone-edit-index').value;
    
    if (editIndex !== '') {
        // Edit existing item
        const index = parseInt(editIndex);
        manualItemsStandalone[index] = {
            material_code: materialCode,
            material_description: materialDesc,
            qty: qty,
            price: price,
            total: total,
            is_manual: true
        };
    } else {
        // Add new item
        manualItemsStandalone.push({
            material_code: materialCode,
            material_description: materialDesc,
            qty: qty,
            price: price,
            total: total,
            is_manual: true
        });
    }
    
    window.manualItemsStandalone = manualItemsStandalone;
    
    renderManualItemsStandalone();
    updateManualItemsSubtotal();
    updateGrandTotal();
    
    // Close modal
    bootstrap.Modal.getInstance(document.getElementById('manualItemModal')).hide();
    
    console.log('✅ Manual standalone items:', manualItemsStandalone);
}

function renderManualItemsStandalone() {
    const container = document.getElementById('manual-items-standalone-container');
    
    if (manualItemsStandalone.length === 0) {
        document.getElementById('no-manual-items-message').style.display = 'block';
        document.getElementById('manual-subtotal-box').style.display = 'none';
        return;
    }
    
    document.getElementById('no-manual-items-message').style.display = 'none';
    document.getElementById('manual-subtotal-box').style.display = 'block';
    
    container.innerHTML = manualItemsStandalone.map((item, index) => `
        <div class="material-card manual">
            <span class="badge-manual">MANUAL</span>
            <div class="d-flex align-items-start gap-3">
                <div class="material-info flex-grow-1">
                    <div class="material-code">${item.material_code}</div>
                    <div class="material-desc">${item.material_description}</div>
                    <div class="material-price">@ ${formatRupiah(item.price)}</div>
                </div>
                <div class="material-qty">
                    <label class="form-label small mb-1">Qty:</label>
                    <div class="fw-bold">${item.qty}</div>
                </div>
                <div class="material-total">
                    <label class="form-label small mb-1">Total:</label>
                    <div class="fw-bold">${formatRupiah(item.total)}</div>
                </div>
                <div class="material-actions">
                    <button type="button" 
                            class="btn btn-sm btn-warning me-1 mt-4" 
                            onclick="editManualItemStandalone(${index})"
                            title="Edit">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button type="button" 
                            class="btn btn-sm btn-danger mt-4" 
                            onclick="removeManualItemStandalone(${index})"
                            title="Hapus">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        </div>
    `).join('');
}

function editManualItemStandalone(index) {
    const item = manualItemsStandalone[index];
    
    document.getElementById('standalone-edit-index').value = index;
    document.getElementById('standalone-material-code').value = item.material_code;
    document.getElementById('standalone-material-desc').value = item.material_description;
    document.getElementById('standalone-material-qty').value = item.qty;
    document.getElementById('standalone-material-price').value = item.price;
    calculateStandaloneMaterialTotal();
    
    const modal = new bootstrap.Modal(document.getElementById('manualItemModal'));
    modal.show();
}

function removeManualItemStandalone(index) {
    if (!confirm('Yakin ingin menghapus item ini?')) return;
    
    manualItemsStandalone.splice(index, 1);
    window.manualItemsStandalone = manualItemsStandalone;
    
    renderManualItemsStandalone();
    updateManualItemsSubtotal();
    updateGrandTotal();
}

function updateManualItemsSubtotal() {
    const subtotal = manualItemsStandalone.reduce((sum, item) => sum + item.total, 0);
    document.getElementById('manual-items-subtotal').textContent = formatRupiah(subtotal);
}

// ============================================
// SEARCH MATERIAL
// ============================================

function searchMaterial(groupId) {
    const searchInput = document.getElementById(`search-material-${groupId}`);
    const searchTerm = searchInput.value.toLowerCase().trim();
    const dropdown = document.getElementById(`search-dropdown-${groupId}`);
    const resultsContainer = document.getElementById(`search-results-${groupId}`);
    
    if (searchTerm.length < 2) {
        resultsContainer.style.display = 'none';
        return;
    }
    
    const type = hardwareGroups[groupId].type;
    const allMaterials = hardwareData[type] || [];
    
    // Filter materials based on search term
    const filteredMaterials = allMaterials.filter(material => {
        const materialName = material.material_description.toLowerCase();
        const materialCode = (material.material_code || '').toLowerCase();
        return materialName.includes(searchTerm) || materialCode.includes(searchTerm);
    });
    
    if (filteredMaterials.length === 0) {
        dropdown.innerHTML = '<option disabled>Tidak ada hasil pencarian</option>';
        resultsContainer.style.display = 'block';
        return;
    }
    
    // Populate dropdown with search results
    dropdown.innerHTML = filteredMaterials.map((material, index) => {
        const originalIndex = allMaterials.findIndex(m => m.id === material.id);
        return `<option value="${originalIndex}">
            ${material.material_code || 'N/A'} - ${material.material_description} - ${formatRupiah(material.price)}
        </option>`;
    }).join('');
    
    resultsContainer.style.display = 'block';
}

// ============================================
// SELECT MATERIAL FROM DROPDOWN
// ============================================

function selectMaterialFromDropdown(groupId) {
    const dropdown = document.getElementById(`search-dropdown-${groupId}`);
    const materialIndex = parseInt(dropdown.value);
    
    if (isNaN(materialIndex)) return;
    
    const type = hardwareGroups[groupId].type;
    const material = hardwareData[type][materialIndex];
    
    // Check if already added
    const alreadyAdded = hardwareGroups[groupId].materials.some(m => m.index === materialIndex && !m.is_manual);
    
    if (alreadyAdded) {
        alert('⚠️ Material ini sudah dipilih!');
        return;
    }
    
    // Add material to selected list
    addSelectedMaterial(groupId, materialIndex, material);
    
    // Clear search
    clearSearch(groupId);
}

// ============================================
// ADD SELECTED MATERIAL TO DISPLAY
// ============================================

function addSelectedMaterial(groupId, materialIndex, material) {
    const selectedContainer = document.getElementById(`selected-materials-${groupId}`);
    
    // Remove "no materials" message if exists
    if (selectedContainer.querySelector('.text-muted')) {
        selectedContainer.innerHTML = '';
    }
    
    const materialCardHtml = `
        <div class="material-card" id="material-card-${groupId}-${materialIndex}">
            <div class="d-flex align-items-start gap-3">
                <div class="material-info flex-grow-1">
                    <div class="material-code">${material.material_code || 'N/A'}</div>
                    <div class="material-desc">${material.material_description}</div>
                    <div class="material-price">@ ${formatRupiah(material.price)}</div>
                </div>
                <div class="material-qty">
                    <label class="form-label small mb-1">Qty:</label>
                    <input type="number" 
                           class="form-control form-control-sm" 
                           id="material-qty-${groupId}-${materialIndex}"
                           placeholder="Qty"
                           min="1"
                           value="1"
                           onchange="updateMaterialQty('${groupId}', ${materialIndex})"
                           style="width: 100px;">
                </div>
                <div class="material-total">
                    <label class="form-label small mb-1">Total:</label>
                    <div class="fw-bold" id="material-total-${groupId}-${materialIndex}">Rp 0</div>
                </div>
                <div class="material-actions">
                    <button type="button" 
                            class="btn btn-sm btn-danger mt-4" 
                            onclick="removeSelectedMaterial('${groupId}', ${materialIndex})"
                            title="Hapus">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        </div>
    `;
    
    selectedContainer.insertAdjacentHTML('beforeend', materialCardHtml);
    
    // Add to materials array
    const materialData = {
        index: materialIndex,
        material_id: material.id,
        material_code: material.material_code,
        material_description: material.material_description,
        qty: 1,
        price: material.price,
        total: material.price,
        is_manual: false
    };
    
    hardwareGroups[groupId].materials.push(materialData);
    window.hardwareGroups = hardwareGroups;
    
    // Update total
    updateMaterialQty(groupId, materialIndex);
}

// ============================================
// REMOVE SELECTED MATERIAL
// ============================================

function removeSelectedMaterial(groupId, materialIndex) {
    if (!confirm('Yakin ingin menghapus material ini?')) return;
    
    // Remove from DOM
    const card = document.getElementById(`material-card-${groupId}-${materialIndex}`);
    if (card) card.remove();
    
    // Remove from array
    hardwareGroups[groupId].materials = hardwareGroups[groupId].materials.filter(m => m.index !== materialIndex);
    window.hardwareGroups = hardwareGroups;
    
    // Check if no materials left
    const selectedContainer = document.getElementById(`selected-materials-${groupId}`);
    if (hardwareGroups[groupId].materials.length === 0) {
        selectedContainer.innerHTML = `
            <div class="text-muted text-center p-3" style="background: #f8f9fa; border-radius: 8px;">
                <i class="fas fa-info-circle me-2"></i>Belum ada material yang dipilih
            </div>
        `;
    }
    
    updateGroupSubtotal(groupId);
}

// ============================================
// CLEAR SEARCH
// ============================================

function clearSearch(groupId) {
    const searchInput = document.getElementById(`search-material-${groupId}`);
    const resultsContainer = document.getElementById(`search-results-${groupId}`);
    
    if (searchInput) searchInput.value = '';
    if (resultsContainer) resultsContainer.style.display = 'none';
}

// ============================================
// UPDATE MATERIAL QTY
// ============================================

function updateMaterialQty(groupId, materialIndex) {
    const qtyInput = document.getElementById(`material-qty-${groupId}-${materialIndex}`);
    if (!qtyInput) return;
    
    const qty = parseInt(qtyInput.value) || 1;
    const type = hardwareGroups[groupId].type;
    const material = hardwareData[type][materialIndex];
    const total = material.price * qty;
    
    document.getElementById(`material-total-${groupId}-${materialIndex}`).textContent = formatRupiah(total);
    
    // Update in materials array
    const materialInArray = hardwareGroups[groupId].materials.find(m => m.index === materialIndex && !m.is_manual);
    if (materialInArray) {
        materialInArray.qty = qty;
        materialInArray.total = total;
        window.hardwareGroups = hardwareGroups;
    }
    
    updateGroupSubtotal(groupId);
}

// ============================================
// UPDATE GROUP SUBTOTAL
// ============================================

function updateGroupSubtotal(groupId) {
    const subtotal = hardwareGroups[groupId].materials.reduce((sum, material) => sum + material.total, 0);
    hardwareGroups[groupId].subtotal = subtotal;
    window.hardwareGroups = hardwareGroups;
    
    document.getElementById(`subtotal-${groupId}`).textContent = formatRupiah(subtotal);
    
    updateGrandTotal();
}

// ============================================
// UPDATE GRAND TOTAL
// ============================================

function updateGrandTotal() {
    // Group subtotals
    const groupsTotal = Object.values(hardwareGroups).reduce((sum, group) => sum + group.subtotal, 0);
    
    // Manual standalone items total
    const manualTotal = manualItemsStandalone.reduce((sum, item) => sum + item.total, 0);
    
    const grandTotal = groupsTotal + manualTotal;
    
    document.getElementById('grand-total-display').textContent = formatRupiah(grandTotal);
    document.getElementById('grand_total').value = grandTotal;
}

// ============================================
// FORMAT RUPIAH
// ============================================

function formatRupiah(amount) {
    return 'Rp ' + Math.round(amount).toLocaleString('id-ID');
}

// ============================================
// SUBMIT FORM
// ============================================

document.getElementById('hardwareForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    // Check if at least one item exists (group OR manual standalone)
    const hasGroupMaterial = Object.values(hardwareGroups).some(group => group.materials.length > 0);
    const hasManualItems = manualItemsStandalone.length > 0;
    
    if (!hasGroupMaterial && !hasManualItems) {
        alert('❌ Minimal harus ada 1 material atau item manual!');
        return;
    }
    
    // ✅ CRITICAL: Create clean copy for submission
    const hardwareDataToSave = {
        groups: [],
        manual_items: [],
        grand_total: parseFloat(document.getElementById('grand_total').value)
    };
    
    // Clean copy groups
    Object.values(hardwareGroups).forEach(group => {
        hardwareDataToSave.groups.push({
            hardware_type: group.type,
            materials: group.materials.map(m => ({
                material_code: m.material_code,
                material_description: m.material_description,
                qty: m.qty,
                price: m.price,
                total: m.total,
                is_manual: m.is_manual || false
            })),
            subtotal: group.subtotal
        });
    });
    
    // Clean copy manual items
    manualItemsStandalone.forEach(item => {
        hardwareDataToSave.manual_items.push({
            material_code: item.material_code,
            material_description: item.material_description,
            qty: item.qty,
            price: item.price,
            total: item.total,
            is_manual: true
        });
    });
    
    console.log('[SUBMIT] Hardware Data to Save:', hardwareDataToSave);
    
    document.getElementById('loadingOverlay').style.display = 'flex';
    
    fetch('{{ route("invoice.hardware.update", $invoiceHardware->id) }}', {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({
            hardware_data: JSON.stringify(hardwareDataToSave),
            grand_total: hardwareDataToSave.grand_total
        })
    })
    .then(response => response.json())
    .then(data => {
        console.log('[SUBMIT] Server Response:', data);
        
        if (data.success) {
            alert('✅ Invoice Hardware berhasil diupdate!');
            window.location.href = '{{ route("invoice.hardware.show", $invoiceHardware->id) }}';
        } else {
            throw new Error(data.message || 'Failed to update invoice hardware');
        }
    })
    .catch(error => {
        console.error('[SUBMIT] Error:', error);
        alert('❌ Terjadi kesalahan: ' + error.message);
    })
    .finally(() => {
        document.getElementById('loadingOverlay').style.display = 'none';
    });
});
</script>

@endsection