<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use App\Models\{User, Role, Company};

class AdminController extends Controller
{
    public function index(Request $request)
    {
        // Halaman admin default: tampilkan daftar user, role, dan company
        $users = User::with(['companies','role'])->orderBy('name')->paginate(20);
        $roles = Role::orderBy('code')->get();
        $companies = Company::orderBy('code')->get();

        return view('admin.index', compact('users', 'roles', 'companies'));
    }

    public function storeUser(Request $r)
    {
        $data = $r->validate([
            'name'          => ['required','string','max:255'],
            'email'         => ['required','email','max:255'],
            'role_id'       => ['nullable','exists:roles,id'],
            'notify_email'  => ['nullable','string','max:500'],
            'company_ids'   => ['nullable','array'],
            'company_ids.*' => ['integer','exists:companies,id'],
        ]);

        // Normalisasi email notifikasi
        $notify = trim((string)($data['notify_email'] ?? ''));
        if ($notify !== '') {
            $parts = preg_split('/[,\s;]+/', $notify, -1, PREG_SPLIT_NO_EMPTY) ?: [];
            $parts = collect($parts)
                ->map(fn($e) => mb_strtolower(trim($e)))
                ->filter(fn($e) => filter_var($e, FILTER_VALIDATE_EMAIL))
                ->unique()
                ->values()
                ->all();
            $notify = implode(', ', $parts);
        } else {
            $notify = null;
        }

        // Upsert user berdasarkan email
        $user = User::firstOrNew(['email' => $data['email']]);
        $user->name = $data['name'];
        $user->role_id = $data['role_id'] ?? null;
        $user->notify_email = $notify;
        $user->save();

        // Sinkronisasi company
        $ids = collect($data['company_ids'] ?? [])->map(fn($v)=>(int)$v)->filter()->all();
        $user->companies()->sync($ids);

        return back()->with('success', 'User added/updated successfully.')->withFragment('users');
    }

    public function updateUser(Request $r, User $user)
    {
        $validated = $r->validate([
            'name'          => ['required','string','max:255'],
            'notify_email'  => ['nullable','string','max:255'],
            'role_id'       => ['nullable','exists:roles,id'],
            'company_ids'   => ['nullable','array'],
            'company_ids.*' => ['integer','exists:companies,id'],
            'company_codes' => ['nullable','string'],
        ]);

        $user->fill(Arr::only($validated, ['name','notify_email','role_id']))->save();

        $ids = collect($validated['company_ids'] ?? [])->filter()->map(fn($v) => (int)$v)->all();

        if (!empty($validated['company_codes'])) {
            $codes = collect(preg_split('/[,\s;]+/', $validated['company_codes'], -1, PREG_SPLIT_NO_EMPTY))
                ->map(fn($c) => strtoupper(trim($c)))->all();
            $byCode = Company::whereIn('code', $codes)->pluck('id')->all();
            $ids = array_values(array_unique(array_merge($ids, $byCode)));
        }

        $user->companies()->sync($ids);

        return back()->with('success', 'User updated successfully.')->withFragment('users');
    }

    public function destroyUser(Request $r, User $user)
    {
        $me = $r->attributes->get('me');
        if ($me && $me->id === $user->id) {
            return back()->withErrors(['You cannot delete your own account.'])->withFragment('users');
        }
        $user->delete();
        return back()->with('success','User deleted successfully.')->withFragment('users');
    }
}
