@extends('layouts.app')

@section('content')
    <div class="container-fluid py-4">

        {{-- ===================== PAGE HEADER ===================== --}}
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h4 class="fw-bold" style="color:#3B3B3B;">Assembly Order Approvals</h4>
            <div class="d-flex gap-2">
                <input type="text" id="searchBox" class="form-control form-control-sm" placeholder="Search recipe...">
                <select id="sortSelect" class="form-select form-select-sm">
                    <option value="due_date">Sort by Due Date</option>
                    <option value="posting_date">Sort by Posting Date</option>
                    <option value="recipe_name">Sort by Recipe Name</option>
                    <option value="status">Sort by Status</option>
                </select>
            </div>
        </div>

        {{-- ===================== CARD GRID LIST ===================== --}}
        <div id="aoListContainer" class="row g-3">
            @foreach ($aoList as $ao)
                <div class="col-md-4 col-lg-3">
                    <div class="card shadow-sm border-0 ao-card h-100" data-status="{{ strtolower($ao['status']) }}">
                        <div class="card-body">
                            {{-- Header --}}
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <span
                                    class="badge 
                            @if ($ao['status'] == 'Pending') bg-warning text-dark 
                            @elseif($ao['status'] == 'Approved') bg-success 
                            @else bg-secondary @endif
                        ">
                                    {{ $ao['status'] }}
                                </span>
                                <small class="text-muted">{{ $ao['document_no'] }}</small>
                            </div>

                            {{-- Content --}}
                            <h6 class="fw-semibold mb-1">{{ $ao['recipe_name'] }}</h6>
                            <div class="text-muted small mb-2">
                                Qty: <strong>{{ $ao['quantity'] }}</strong> {{ $ao['uom'] }}<br>
                                By: {{ $ao['requested_by'] }}
                            </div>

                            <div class="d-flex justify-content-between small text-muted">
                                <div>
                                    <i class="bi bi-calendar-event me-1"></i> {{ $ao['posting_date'] }}
                                </div>
                                <div>
                                    <i class="bi bi-hourglass-split me-1"></i> Due: {{ $ao['due_date'] }}
                                </div>
                            </div>
                        </div>

                        {{-- Footer (Action Buttons) --}}
                        <div class="card-footer bg-transparent border-0 d-flex justify-content-between">
                            <button class="btn btn-sm btn-outline-primary w-50 me-1 approve-btn">
                                <i class="bi bi-check2-circle"></i> Approve
                            </button>
                            <button class="btn btn-sm btn-outline-danger w-50 reject-btn">
                                <i class="bi bi-x-circle"></i> Reject
                            </button>
                        </div>
                    </div>
                </div>
            @endforeach
        </div>
    </div>
@endsection


{{-- ===================== STYLES ===================== --}}
@push('styles')
    <script>
        document.addEventListener('DOMContentLoaded', async () => {

            const container = document.getElementById('aoListContainer');
            const searchBox = document.getElementById('searchBox');
            const sortSelect = document.getElementById('sortSelect');

            let aoData = [];

            // =====================
            // Loading Skeleton
            // =====================
            function showSkeleton(count = 6) {
                container.innerHTML = '';
                for (let i = 0; i < count; i++) {
                    const skeleton = document.createElement('div');
                    skeleton.className = 'col-md-4 col-lg-3';
                    skeleton.innerHTML = `
                <div class="card shadow-sm border-0 ao-card h-100 p-3">
                    <div class="skeleton shimmer mb-2" style="height:20px; width:40%; border-radius:4px;"></div>
                    <div class="skeleton shimmer mb-1" style="height:18px; width:60%; border-radius:4px;"></div>
                    <div class="skeleton shimmer mb-1" style="height:18px; width:30%; border-radius:4px;"></div>
                    <div class="skeleton shimmer" style="height:14px; width:50%; border-radius:4px;"></div>
                </div>
            `;
                    container.appendChild(skeleton);
                }
            }

            function hideSkeleton() {
                container.innerHTML = '';
            }

            // =====================
            // Fetch Data dari API
            // =====================
            async function fetchAOData() {
                showSkeleton();
                try {
                    const res = await fetch('{{ route('getassemblyorder') }}');
                    if (!res.ok) throw new Error(`HTTP ${res.status}`);
                    const result = await res.json();

                    // Ambil value array
                    const data = result.data ?? [];

                    aoData = data.map(item => ({
                        docNo: item.docNo,
                        recipe_name: item.description,
                        quantity: item.Quantity,
                        postingDate: item.postingDate,
                        dueDate: item.dueDate,
                        status: item.status
                    }));

                    hideSkeleton();
                    filterAndSort();
                } catch (err) {
                    console.error(err);
                    container.innerHTML =
                        `<div class="text-center text-danger py-3">Failed to load Assembly Orders</div>`;
                }
            }

            // =====================
            // Render Card Grid
            // =====================
            function renderCards(data) {
                container.innerHTML = '';
                data.forEach(ao => {
                    const statusClass = ao.status.toLowerCase() === 'released' ? 'bg-success' :
                        'bg-warning text-dark';
                    const card = document.createElement('div');
                    card.className = 'col-md-4 col-lg-3';
                    card.innerHTML = `
                <div class="card shadow-sm border-0 ao-card h-100" data-status="${ao.status.toLowerCase()}" data-doc="${ao.docNo}">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center mb-2">
                            <span class="badge ${statusClass}">${ao.status}</span>
                            <small class="text-muted">${ao.docNo}</small>
                        </div>
                        <h6 class="fw-semibold mb-1">${ao.recipe_name}</h6>
                        <div class="text-muted small mb-2">
                            Qty: <strong>${ao.quantity}</strong>
                        </div>
                        <div class="d-flex justify-content-between small text-muted">
                            <div><i class="bi bi-calendar-event me-1"></i> ${ao.postingDate}</div>
                            <div><i class="bi bi-hourglass-split me-1"></i> Due: ${ao.dueDate}</div>
                        </div>
                    </div>
                    <div class="card-footer bg-transparent border-0 d-flex justify-content-between">
                        <button class="btn btn-sm btn-outline-primary w-50 me-1 approve-btn">
                            <i class="bi bi-check2-circle"></i> Approve
                        </button>
                        <button class="btn btn-sm btn-outline-danger w-50 reject-btn">
                            <i class="bi bi-x-circle"></i> Reject
                        </button>
                    </div>
                </div>
            `;
                    container.appendChild(card);
                });
            }

            // =====================
            // Search & Multi-Level Sort
            // =====================
            function filterAndSort() {
                const query = searchBox.value.toLowerCase();

                let filtered = aoData.filter(ao => ao.recipe_name.toLowerCase().includes(query));

                filtered.sort((a, b) => {
                    const postA = new Date(a.postingDate);
                    const postB = new Date(b.postingDate);

                    // Posting date terbaru dulu (desc)
                    if (postA < postB) return 1;
                    if (postA > postB) return -1;

                    // Kalau posting date sama, urutkan due date lebih cepat dulu (asc)
                    const dueA = new Date(a.dueDate);
                    const dueB = new Date(b.dueDate);
                    return dueA - dueB;
                });

                renderCards(filtered);
            }

            searchBox.addEventListener('input', filterAndSort);
            sortSelect.addEventListener('change', filterAndSort);

            // =====================
            // Approve / Reject Logic
            // =====================
            container.addEventListener('click', async (e) => {
                const approveBtn = e.target.closest('.approve-btn');
                const rejectBtn = e.target.closest('.reject-btn');

                // =========================
                // APPROVE / POST AO
                // =========================
                if (approveBtn) {
                    const card = approveBtn.closest('.ao-card');
                    const AssemblyNo = card.dataset.doc;

                    const confirm = await Swal.fire({
                        title: `Post Assembly Order ${AssemblyNo}?`,
                        text: 'Pastikan semua data sudah benar sebelum posting.',
                        icon: 'question',
                        showCancelButton: true,
                        confirmButtonText: 'Ya, Post Sekarang',
                        cancelButtonText: 'Batal',
                        reverseButtons: true,
                        confirmButtonColor: '#a47148',
                        cancelButtonColor: '#d3d3d3'
                    });

                    if (!confirm.isConfirmed) return;

                    approveBtn.disabled = true;
                    approveBtn.innerHTML =
                        `<span class="spinner-border spinner-border-sm me-1"></span>Posting...`;

                    try {
                        const res = await fetch(`{{ route('bc.postAssembly') }}`, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': '{{ csrf_token() }}'
                            },
                            body: JSON.stringify({
                                AssemblyNo
                            })
                        });

                        const result = await res.json();

                        // =======================
                        // SUCCESS
                        // =======================
                        if (res.ok && result.status === 'success') {
                            Swal.fire({
                                icon: 'success',
                                title: 'Assembly Posted!',
                                text: `Assembly Order ${AssemblyNo} berhasil diposting.`,
                                showConfirmButton: false,
                                timer: 1000
                            });

                            // update badge & button
                            card.querySelector('.badge').className = 'badge bg-success';
                            card.querySelector('.badge').textContent = 'Released';
                            card.dataset.status = 'released';
                            approveBtn.innerHTML = `<i class="bi bi-check2-circle"></i> Posted`;

                            setTimeout(() => location.reload(), 1000);
                        }

                        // =======================
                        // FAILED (dengan detail)
                        // =======================
                        else {
                            let errorMsg = result.message || 'Gagal memproses request.';

                            // Jika ada field "response" → ambil message dari JSON-nya
                            if (result.response) {
                                try {
                                    const parsed = JSON.parse(result.response);
                                    if (parsed?.error?.message) {
                                        errorMsg = parsed.error.message;
                                    }
                                } catch (e) {
                                    console.warn('Failed to parse response JSON:', e);
                                }
                            }

                            Swal.fire({
                                icon: 'error',
                                title: 'Gagal Posting Assembly!',
                                html: `<pre style="text-align:left;white-space:pre-wrap;">${errorMsg}</pre>`,
                                confirmButtonColor: '#a47148'
                            });

                            approveBtn.innerHTML = `<i class="bi bi-check2-circle"></i> Approve`;
                            approveBtn.disabled = false;
                        }
                    }

                    // =======================
                    // EXCEPTION HANDLER
                    // =======================
                    catch (error) {
                        console.error(error);
                        let errorMsg = error.message || 'Terjadi kesalahan tak terduga.';

                        // Jika response error dari fetch berupa JSON/text
                        if (error instanceof Response) {
                            try {
                                const errJson = await error.json();
                                errorMsg = errJson.message || JSON.stringify(errJson);
                            } catch {
                                errorMsg = await error.text();
                            }
                        }

                        Swal.fire({
                            icon: 'error',
                            title: 'Server Error',
                            html: `<pre style="text-align:left;white-space:pre-wrap;">${errorMsg}</pre>`,
                            confirmButtonColor: '#a47148'
                        });

                        approveBtn.innerHTML = `<i class="bi bi-check2-circle"></i> Approve`;
                        approveBtn.disabled = false;
                    }
                }

                // =========================
                // REJECT AO (REOPEN)
                // =========================
                if (rejectBtn) {
                    const card = rejectBtn.closest('.ao-card');

                    card.querySelector('.badge').className = 'badge bg-warning text-dark';
                    card.querySelector('.badge').textContent = 'Open';
                    card.dataset.status = 'open';

                    Swal.fire({
                        icon: 'info',
                        title: 'Assembly Order dibuka kembali',
                        text: 'Status telah diubah menjadi Open.',
                        timer: 1000,
                        showConfirmButton: false
                    });
                }
            });


            // =====================
            // Init Load
            // =====================
            await fetchAOData();

        });
    </script>
@endpush


@push('styles')
    <style>
        .skeleton {
            background-color: #e2e2e2;
            position: relative;
            overflow: hidden;
        }

        .shimmer::before {
            content: '';
            display: block;
            position: absolute;
            top: 0;
            left: -150px;
            height: 100%;
            width: 150px;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.4), transparent);
            animation: shimmer 1.5s infinite;
        }

        @keyframes shimmer {
            0% {
                left: -150px;
            }

            100% {
                left: 100%;
            }
        }
    </style>
@endpush
