@extends('layouts.app')

@section('title', 'Inventory Lot')

@section('content')
    <div class="container py-4">
        <div class="d-flex align-items-center mb-3 flex-wrap gap-2">
            <h3 class="fw-bold text-brown mb-0">Inventory</h3>
            <div class="ms-auto d-flex align-items-center gap-2 w-100 w-md-auto">
                <input id="soSearch" class="form-control form-control-sm flex-grow-1"
                    placeholder="Cari no / description / to-code ...">
                <select id="sortSelect" class="form-select form-select-sm w-auto">
                    <option value="default">Urutkan</option>
                    <option value="no-asc">No ↑</option>
                    <option value="no-desc">No ↓</option>
                    <option value="date-asc">Tanggal ↑</option>
                    <option value="date-desc">Tanggal ↓</option>
                    <option value="lines-desc">Jumlah Lines ↓</option>
                </select>
                <button id="refreshBtn" class="btn btn-sm btn-outline-brown">Refresh</button>
            </div>
        </div>

        <div id="soCount" class="text-muted small mb-3"></div>
        <div id="soList" class="row g-3"></div>
        <nav id="paginationWrap" aria-label="SO pagination" class="mt-3"></nav>
    </div>
@endsection
@push('scripts')
    <script>
document.addEventListener('DOMContentLoaded', () => {

    const listEl = document.getElementById('soList');
    const countEl = document.getElementById('soCount');
    const searchInput = document.getElementById('soSearch');
    const sortSelect = document.getElementById('sortSelect');
    const refreshBtn = document.getElementById('refreshBtn');

    // DATA DARI CONTROLLER
    const dataStore = @json($items);
    let filtered = [...dataStore]; // clone

    // ===== SORT DEFAULT (ItemNo ASC) =====
    filtered.sort((a, b) => (a.ItemNo || "").localeCompare(b.ItemNo || ""));
    renderList(true); // ← RENDER AWAL

    // ==========================================================
    // 🔍 SEARCH HANDLER
    // ==========================================================
    searchInput.addEventListener('input', () => {
        applyFilterSort();
    });

    // ==========================================================
    // SORT HANDLER
    // ==========================================================
    sortSelect.addEventListener('change', () => {
        applyFilterSort();
    });

    // ==========================================================
    // REFRESH
    // ==========================================================
    refreshBtn.addEventListener('click', () => {
        searchInput.value = "";
        sortSelect.value = "default";
        filtered = [...dataStore];

        filtered.sort((a, b) => (a.ItemNo || "").localeCompare(b.ItemNo || ""));
        renderList(true); // ← dianggap fresh, popup akan muncul
    });

    // ==========================================================
    // FILTER + SORT
    // ==========================================================
    function applyFilterSort() {
        const keyword = searchInput.value.toLowerCase().trim();
        const sortMode = sortSelect.value;

        filtered = dataStore.filter(item => {
            const no = (item.ItemNo || "").toLowerCase();
            const desc = (item.ItemDescription || "").toLowerCase();
            const hasLot = (item.lots || []).some(l =>
                (l.Lot_No || "").toLowerCase().includes(keyword)
            );

            return (
                no.includes(keyword) ||
                desc.includes(keyword) ||
                hasLot
            );
        });

        switch (sortMode) {
            case "no-asc":
                filtered.sort((a, b) => (a.ItemNo || "").localeCompare(b.ItemNo || ""));
                break;

            case "no-desc":
                filtered.sort((a, b) => (b.ItemNo || "").localeCompare(a.ItemNo || ""));
                break;

            case "date-asc":
                filtered.sort((a, b) => getNearestExpiry(a) - getNearestExpiry(b));
                break;

            case "date-desc":
                filtered.sort((a, b) => getNearestExpiry(b) - getNearestExpiry(a));
                break;

            case "lines-desc":
                filtered.sort((a, b) => (b.lots?.length || 0) - (a.lots?.length || 0));
                break;
        }

        renderList(false); // ← search/sort: popup TIDAK muncul
    }

    // ==========================================================
    // GET NEAREST EXPIRY
    // ==========================================================
    function getNearestExpiry(item) {
        if (!item.lots) return Infinity;

        const today = new Date();
        let nearest = Infinity;

        item.lots.forEach(lot => {
            if (!lot.Expiration_Date) return;
            const exp = new Date(lot.Expiration_Date);
            const diff = (exp - today) / 86400000;
            if (diff > 0 && diff < nearest) nearest = diff;
        });

        return nearest;
    }

    // ==========================================================
    // RENDER LIST
    // ==========================================================
    function renderList(isFirstLoad = false) {
        const total = filtered.length;
        countEl.textContent = `${total} items`;
        listEl.innerHTML = '';

        const alertQueue = [];

        filtered.forEach((item, idx) => {
            const collapseId = `collapse-${idx}`;
            const card = document.createElement('div');
            card.className = 'col-12 fade-slide show';

            const cleanLots = (item.lots || []).filter(lot => {
                const ln = lot.Lot_No ?? '';
                return ln.trim() !== '';
            });

            cleanLots.sort((a, b) => {
                let da = new Date(a.Expiration_Date || '1970-01-01');
                let db = new Date(b.Expiration_Date || '1970-01-01');
                return db - da;
            });

            let nearestExpiry = "-";
            const today = new Date();
            let nearestDiff = Infinity;

            cleanLots.forEach(lot => {
                if (!lot.Expiration_Date) return;

                const exp = new Date(lot.Expiration_Date);
                const diffDays = (exp - today) / 86400000;

                if (diffDays > 0 && diffDays < nearestDiff) {
                    nearestDiff = diffDays;
                    nearestExpiry = lot.Expiration_Date;
                }
            });

            if (nearestExpiry !== "-" && nearestDiff <= 7 && nearestDiff >= 0) {
                alertQueue.push({
                    no: item.ItemNo,
                    desc: item.ItemDescription,
                    date: nearestExpiry
                });
            }

            card.innerHTML = `
<div class='so-card p-0'>
    <div class='card-head'>
        <div class='d-flex justify-content-between align-items-center w-100'>
            <div class="me-3">
                <div class='fw-semibold' style='font-size:1.05rem'>
                    ${escapeHtml(item.ItemNo)}
                </div>
                <div class='text-muted small'>
                    ${escapeHtml(item.ItemDescription)}
                </div>
                <div class="small mt-1 ${nearestExpiry !== '-' ? 'text-danger fw-bold' : 'text-muted'}">
                    Exp. Terdekat : ${nearestExpiry !== '-' ? escapeHtml(nearestExpiry) : '-'}
                </div>
            </div>

            <div class="text-end">
                <div class="fw-bold">Quantity</div>
                <div>
                    <span class='small text-muted'>${item.TotalQty}</span>
                    <span class='small text-muted'>${escapeHtml(item.BaseUoM)}</span>
                </div>

                <button class='btn btn-sm btn-outline-brown mt-2'
                        data-bs-toggle='collapse'
                        data-bs-target='#${collapseId}'>
                    Detail ▾
                </button>
            </div>
        </div>
    </div>

    <div id='${collapseId}' class='collapse'>
        <div class='p-3'>
            <div class='table-responsive'>
                <table class='table table-sm small align-middle'>
                    <thead>
                        <tr>
                            <th>Lot No</th>
                            <th>Expiration</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${ cleanLots.map(lot => `
                            <tr>
                                <td>${escapeHtml(lot.Lot_No)}</td>
                                <td>${escapeHtml(lot.Expiration_Date ?? '-')}</td>
                            </tr>
                        `).join('') }
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
`;

            listEl.appendChild(card);
        });

        // ======================================================
        // 🔔 POPUP HANYA MUNCUL KETIKA HALAMAN MURNI BARU
        // ======================================================
        if (
            isFirstLoad &&
            searchInput.value.trim() === "" &&
            sortSelect.value === "default"
        ) {
            showExpiryAlerts(alertQueue);
        }
    }

    // ======================================================
    // POPUP HANDLER
    // ======================================================
    function showExpiryAlerts(queue) {
        if (!queue.length) return;

        let index = 0;

        function nextAlert() {
            if (index >= queue.length) return;

            const item = queue[index];
            index++;

            Swal.fire({
                icon: "warning",
                title: `Expiring Soon (H-7)!`,
                html: `
                    <div style="text-align:left">
                        <b>Item:</b> ${escapeHtml(item.no)}<br>
                        <b>Description:</b> ${escapeHtml(item.desc)}<br>
                        <b>Nearest Exp:</b> ${escapeHtml(item.date)}
                    </div>
                `,
                confirmButtonText: "Next"
            }).then(() => nextAlert());
        }

        nextAlert();
    }

    // =======================
    function escapeHtml(s) {
        return String(s || '').replace(/[&<>]/g, c => ({
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;'
        }[c]));
    }

    function formatNumber(n) {
        return n;
    }

});
</script>





    @push('styles')
        <style>
            :root {
                --brown-soft: #A67856;
                --brown-soft-2: #D8B79A;
                --cream-bg: #fff9f4;
                --muted: #7a6a62;
            }

            body {
                background: var(--cream-bg);
            }

            /* dark theme adaptation */
            [data-bs-theme="dark"] body {
                background-color: #1e1c1a !important;
            }

            [data-bs-theme="dark"] .so-card {
                background: #2a2623 !important;
                border-color: rgba(255, 255, 255, 0.05) !important;
            }

            [data-bs-theme="dark"] .text-brown,
            [data-bs-theme="dark"] .kv,
            [data-bs-theme="dark"] .fw-semibold {
                color: #f5ede8 !important;
            }

            /* card */
            .so-card {
                background: linear-gradient(180deg, rgba(255, 255, 255, 0.98), rgba(255, 250, 245, 0.98));
                border-radius: 14px;
                box-shadow: 0 8px 24px rgba(103, 76, 62, 0.08);
                border: 1px solid rgba(166, 136, 116, 0.10);
                overflow: hidden;
                transition: transform .28s cubic-bezier(.2, .9, .2, 1), box-shadow .28s ease;
            }

            .so-card:hover {
                transform: translateY(-6px) scale(1.002);
                box-shadow: 0 18px 40px rgba(103, 76, 62, 0.12);
            }

            /* header strip */
            .so-card .card-head {
                padding: .9rem 1rem;
                display: flex;
                align-items: center;
                justify-content: space-between;
                flex-wrap: wrap;
                gap: 12px;
                border-bottom: 1px solid rgba(166, 136, 116, 0.06);
                background: linear-gradient(90deg, rgba(166, 136, 116, 0.04), rgba(216, 183, 154, 0.02));
            }

            /* small details */
            .so-meta {
                font-size: .86rem;
                color: var(--muted);
            }

            .badge-brown {
                background: rgba(166, 136, 116, 0.12);
                color: var(--brown-soft);
                border-radius: 999px;
                padding: .25rem .6rem;
                font-weight: 600;
            }

            /* animated caret */
            .caret-anim {
                transition: transform .32s cubic-bezier(.2, .9, .2, 1);
            }

            /* subtle progressive reveal */
            .fade-slide {
                opacity: 0;
                transform: translateY(8px);
                transition: opacity .36s ease, transform .36s cubic-bezier(.2, .9, .2, 1);
            }

            .fade-slide.show {
                opacity: 1;
                transform: translateY(0);
            }

            /* table */
            .table.small {
                font-size: .86rem;
            }

            /* page controls */
            .btn-outline-brown {
                color: var(--brown-soft);
                border-color: rgba(166, 136, 116, 0.18);
            }

            .btn-outline-brown:hover {
                background: rgba(166, 136, 116, 0.06);
            }

            /* small utilities */
            .kv {
                color: var(--muted);
                font-size: .9rem;
            }

            /* modal theme */
            .modal-theme {
                background: linear-gradient(90deg, rgba(150, 100, 60, 0.06), rgba(255, 240, 230, 0.03));
            }

            /* tablet friendly layout */
            @media (max-width: 992px) {
                .so-card .card-head {
                    flex-direction: column;
                    align-items: flex-start;
                }

                .so-card .p-3 {
                    font-size: 0.9rem;
                }

                #soSearch {
                    width: 100%;
                }
            }
        </style>
        <style>
            /* Tombol Ship Selected - floating style */
            #shipSelectedBtn {
                position: fixed;
                bottom: 25px;
                right: 35px;
                background-color: #a47148;
                /* warm brown */
                color: #fff;
                border: none;
                border-radius: 50px;
                padding: 12px 24px;
                font-weight: 600;
                box-shadow: 0 4px 10px rgba(0, 0, 0, 0.2);
                transition: all 0.3s ease;
                z-index: 999;
            }

            #shipSelectedBtn:hover {
                background-color: #8c5e3c;
                transform: translateY(-2px);
                box-shadow: 0 6px 14px rgba(0, 0, 0, 0.25);
            }

            #shipSelectedBtn:disabled {
                opacity: 0.6;
                cursor: not-allowed;
            }
        </style>
    @endpush
@endpush
