<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\BusinessCentralService;
use App\Services\InventoryBC;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Promise;
use GuzzleHttp\Promise\Create;
use GuzzleHttp\Promise\Utils;
use GuzzleHttp\Client;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Illuminate\Support\Str;
use GuzzleHttp\Exception\RequestException;

class BusinessCentralController extends Controller
{   

    public function getAccessToken()
    {
        $client = new Client();
        $response = $client->post("https://login.microsoftonline.com/".env('AZURE_TENANT_ID')."/oauth2/v2.0/token", [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => env('AZURE_CLIENT_ID'),
                'client_secret' => env('AZURE_CLIENT_SECRET'),
                'scope' => 'https://api.businesscentral.dynamics.com/.default'
            ]
        ]);
        $body = json_decode((string)$response->getBody(), true);

        return $body['access_token'];
    }

    public function getCompanyId($token)
    {
        $client = new Client();
        $response = $client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies", [
            'headers' => [
                'Authorization' => "Bearer {$token}",
                'Accept'        => 'application/json'
            ]
        ]);
        $companies = json_decode((string)$response->getBody(), true);
        $role = session('user')['role'] ?? null; 
        $companyId1 = $companies['value'][0]['id'];
        $companyId2 = $companies['value'][1]['id'];

        return [
            'companyId1' => $companyId1,
            'companyId2' => $companyId2
        ];
    }

    public function __construct()
    {
     $this->token = $this->getAccessToken();
     $companyIds = $this->getCompanyId($this->token);
     $this->companyRegent = $companyIds['companyId1'];
     $this->companyHIN    = $companyIds['companyId2'];
     
     $activeCompany = session('current_company_name') ?? session('user')['role'];
    if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
        $this->companyId = $this->companyRegent;
    } elseif ($activeCompany === 'HIN') {
        $this->companyId = $this->companyHIN;
    }
     $this->client = new Client();
    }


    public function createAndPostPO()
    {
        $bc = new BusinessCentralService();
        $purchaseOrder= $bc->CreatePurchaseOrder();
        return response()->json($purchaseOrder);
    }

    public function showPoSuggestions(Request $request)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;

        $service = new BusinessCentralService();

        $cacheKey = 'po_suggestions_' . md5($userName);
        if (Cache::has($cacheKey)) {
            Log::info("CACHE HIT: $cacheKey");
        } else {
            Log::info("CACHE MISS: $cacheKey");
        }
        $poData = Cache::remember($cacheKey, now()->addMinutes(480), function () use ($userName) {
            $service = new BusinessCentralService();
            return $service->getPoSuggestions($userName);
        });

        $allItems = $poData['items'];
        $vendorMap = $poData['vendors'];
        $vendorOptionsByItemNo = [];
        $uniqueItems = [];
        foreach ($allItems as $item) {
            $itemNo = $item['item_no'] ?? null;
            $vendorNo = $item['vendor_no'] ?? null;
            $vendorName = $item['vendor_name'] ?? null;
            $unitCost = $item['unit_cost'] ?? null;
            if ($vendorNo && $vendorName) {
                $vendorOptionsByItemNo[$itemNo][$vendorNo] = [
                    'name' => $vendorName,
                    'unit_cost' => $unitCost,
                ];
            } elseif ($vendorNo === '0') {
                $vendorOptionsByItemNo[$itemNo][$vendorNo] = [
                    'name' => 'No Vendor',
                    'unit_cost' => 0,
                ];
            }

            if (!isset($uniqueItems[$itemNo])) {
                $uniqueItems[$itemNo] = $item;
            }
        }

        $vendorSet = collect($allItems)
            ->filter(fn ($item) => $item['vendor_no'] && $item['vendor_name'])
            ->pluck('vendor_name', 'vendor_no')
            ->unique()
            ->sortBy(fn ($name, $no) => $name)
            ->toArray();

        $locationSet = collect($allItems)->pluck('location_code')->filter()->unique()->sort()->values()->all();

        $statusSet = collect($allItems)
        ->pluck('status')
        ->filter()
        ->unique()
        ->sort()
        ->values()
        ->all();

        $statusItemSet = collect($allItems)
        ->pluck('Status')
        ->filter()
        ->unique()
        ->sort()
        ->values()
        ->all();

        
        session([
            'po_items' => array_values($uniqueItems),
            'vendor_map' => $vendorMap
        ]);

        return view('po-suggestion', [
            'items'                => array_values($uniqueItems),
            'vendors'              => $vendorSet,
            'locations'            => $locationSet,
            'vendorOptionsByItem'  => $vendorOptionsByItemNo,
            'statuses'             => $statusSet,
            'statusItems'           => $statusItemSet
        ]);
    }

    public function refreshPoSuggestions(Request $request)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;

        $cacheKey = 'po_suggestions_' . md5($userName);
        Cache::forget($cacheKey);

        return redirect()->route('po-suggestions')->with('success', 'Cache has been refreshed.');
    }

    public function refreshMinMaxSuggestions(Request $request)
    {
        Cache::forget('sku_mapping_cache');
       return redirect()->route('sku-mapping')->with('success', 'Cache has been refreshed.');
    }


    public function viewSkuMapping()
    {
        $start = microtime(true);

        $bcService = new InventoryBC(); // adjust if injected
        $data = $bcService->getItemLedgerAndSkuMappingAsync();

        $executionTime = round(microtime(true) - $start, 2);
        
        $currentCompany = session('current_company_name', session('user')['role'] ?? '');

        if ($currentCompany === 'Regent' || $currentCompany === 'SUPER') {
            $locationOptions = ['CI.1010', 'CI.1011' , 'CI.1020'];
        } elseif ($currentCompany === 'HIN') {
            $locationOptions = ['HIN.1200', 'HIN.1300', 'HIN.1000', 'HIN.3000'];
        }
        \Log::info("viewSkuMapping executed in {$executionTime} seconds");
        return view('sku-mapping', [
            'result' => $data['result'],
            'executionTime' => $executionTime,
            'locationOptions' => $locationOptions
        ]);
    }
    
    public function exportSkuMapping(Request $request)
    {
        $bcService = new InventoryBC();
        $data = $bcService->getItemLedgerAndSkuMappingAsync();
        $rows = collect($data['result'] ?? []);
        $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
        $headers = [
            'Authorization' => "Bearer {$this->token}",
            'Accept'        => 'application/json',
            'Prefer'        => 'odata.maxpagesize=20000',
        ];

        $today = now()->toDateString();
        $twoMonthsAgo = now()->subMonths(2)->toDateString();

        $pricePromise = $this->client->getAsync(
            "$baseOdata/Price_List_Lines?" .
            '$filter=' . urlencode("SourceNo ne null and StartingDate le $today and EndingDate ge $twoMonthsAgo") .
            '&$select=Asset_No,Product_No,Unit_of_Measure_Code,DirectUnitCost,SourceNo',
            ['headers' => $headers]
        );
        $uomPromise = $this->client->getAsync(
            "$baseOdata/Item_UOM?" .
            '$select=Item_No,Code,Qty_per_Unit_of_Measure',
            ['headers' => $headers]
        );

        $priceResp = $pricePromise->wait();
        $uomResp   = $uomPromise->wait();

        $priceLines = json_decode((string) $priceResp->getBody(), true)['value'] ?? [];
        $itemUoms   = json_decode((string) $uomResp->getBody(), true)['value'] ?? [];

        // ---- UoM map per item + detect Base UoM (qty == 1 preferred) ----
        $uomMapByItem = []; // [itemNo => ['codes'=>[uom=>qty], 'base'=>uom]]
        foreach ($itemUoms as $u) {
            $item = $u['Item_No'] ?? null;
            $code = $u['Code'] ?? null;
            $qty  = (float)($u['Qty_per_Unit_of_Measure'] ?? 0);
            if (!$item || !$code) continue;

            if (!isset($uomMapByItem[$item])) $uomMapByItem[$item] = ['codes'=>[], 'base'=>null];
            $uomMapByItem[$item]['codes'][$code] = $qty;
            if ($qty == 1.0 && $uomMapByItem[$item]['base'] === null) {
                $uomMapByItem[$item]['base'] = $code;
            }
        }
        foreach ($uomMapByItem as $item => &$info) {
            if ($info['base'] === null && !empty($info['codes'])) {
                $info['base'] = array_key_first($info['codes']);
            }
        }
        unset($info);

        $bestPriceByItem = []; 
        foreach ($priceLines as $pl) {
            $itemNo   = $pl['Asset_No'] ?? ($pl['Product_No'] ?? null);
            if (!$itemNo) continue;

            $priceUom = $pl['Unit_of_Measure_Code'] ?? null;
            $direct   = (float)($pl['DirectUnitCost'] ?? 0);
            if ($direct <= 0) continue;

            $uomInfo  = $uomMapByItem[$itemNo] ?? null;
            if (!$uomInfo) continue;

            $baseUom  = $uomInfo['base'] ?? null;
            $qtyBase  = (float)($baseUom ? ($uomInfo['codes'][$baseUom] ?? 1) : 1);
            $qtyPrice = (float)($priceUom ? ($uomInfo['codes'][$priceUom] ?? 0) : 0);
            if (!$baseUom || $qtyPrice <= 0) continue;

            $pricePerBase = $direct / $qtyPrice * $qtyBase; // normalize to base uom

            if (!isset($bestPriceByItem[$itemNo]) || $pricePerBase < $bestPriceByItem[$itemNo]['price']) {
                $bestPriceByItem[$itemNo] = [
                    'price'    => $pricePerBase,
                    'priceUom' => $priceUom,
                    'baseUom'  => $baseUom,
                ];
            }
        }

        $location     = $request->query('location');
        $search       = mb_strtolower(trim($request->query('search', '')));
        $statusFilter = mb_strtolower(trim($request->query('status', '')));
        $selectedKeys = $request->input('selected', []);

        $rows = $rows->map(function ($r) {
            if (!isset($r['Consumption'])) {
                $r['Consumption'] = $r['Consumption'] ?? ($r['Consumption / 2 months'] ?? 0);
            }
            return $r;
        });

        if (!empty($location)) {
            $rows = $rows->filter(fn ($r) => ($r['Location'] ?? '') === $location);
        }
        if ($search !== '') {
            $rows = $rows->filter(function ($r) use ($search) {
                $item  = mb_strtolower($r['Item_No'] ?? '');
                $desc  = mb_strtolower($r['Description'] ?? '');
                return str_contains($item, $search) || str_contains($desc, $search);
            });
        }
        if ($statusFilter !== '') {
            $rows = $rows->filter(function ($r) use ($statusFilter) {
                $isActive = (float)($r['Consumption'] ?? 0) !== 0.0;
                return $statusFilter === 'active' ? $isActive : !$isActive;
            });
        }
        if (!empty($selectedKeys) && is_array($selectedKeys)) {
            $keySet = collect($selectedKeys)->flip();
            $rows = $rows->filter(function ($r) use ($keySet) {
                $k = ($r['Item_No'] ?? '') . '|' . ($r['Location'] ?? '');
                return $keySet->has($k);
            });
        }
        $rows = $rows->values();

        $fileName = 'sku-mapping_' . now()->format('Ymd_His') . '.xlsx';

        return \Maatwebsite\Excel\Facades\Excel::download(
            new class($rows, $bestPriceByItem, $uomMapByItem) implements
                \Maatwebsite\Excel\Concerns\FromCollection,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\WithMapping,
                \Maatwebsite\Excel\Concerns\WithColumnFormatting,
                \Maatwebsite\Excel\Concerns\WithStyles,
                \Maatwebsite\Excel\Concerns\WithEvents,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $rows;
                private $bestPriceByItem;
                private $uomMapByItem;

                public function __construct($rows, $bestPriceByItem, $uomMapByItem) {
                    $this->rows = $rows;
                    $this->bestPriceByItem = $bestPriceByItem;
                    $this->uomMapByItem = $uomMapByItem;
                }

                public function headings(): array
                {
                    return [
                        'Item No',
                        'Description',
                        'Location',
                        'Stock For (Days)',
                        'Lead Time',
                        'Consumption (2 Months)',

                        'Suggest Min',
                        'Temp Min',
                        'Actual Min',
                        'Forecast Min',

                        'Suggest Max',
                        'Temp Max',
                        'Actual Max',
                        'Forecast Max',

                        'Price / Base UoM',

                        'Cost (Suggest Min)',
                        'Cost (Temp Min)',
                        'Cost (Actual Min)',
                        'Cost (Forecast Min)',

                        'Cost (Suggest Max)',
                        'Cost (Temp Max)',
                        'Cost (Actual Max)',
                        'Cost (Forecast Max)',

                        'Base UoM',
                        'Price UoM (Listed)',
                    ];
                }

                public function collection()
                {
                    return $this->rows;
                }

                public function map($r): array
                {
                    $itemNo = $r['Item_No'] ?? '';

                    $priceInfo = $this->bestPriceByItem[$itemNo] ?? null;
                    $priceBase = isset($priceInfo['price']) ? (float)$priceInfo['price'] : null;
                    $baseUom   = $priceInfo['baseUom']  ?? ($this->uomMapByItem[$itemNo]['base'] ?? '');
                    $priceUom  = $priceInfo['priceUom'] ?? '';

                    // Quantities (ensure floats)
                    $sMin = (float)($r['SuggestMinQty'] ?? 0);
                    $tMin = (float)($r['ActMinQty'] ?? 0);
                    $aMin = (float)($r['RealMinQty'] ?? 0);
                    $fMin = (float)($r['ForecMin'] ?? 0);

                    $sMax = (float)($r['SuggestMaxQty'] ?? 0);
                    $tMax = (float)($r['ActMaxQty'] ?? 0);
                    $aMax = (float)($r['RealMaxQty'] ?? 0);
                    $fMax = (float)($r['ForecMax'] ?? 0);

                    $cost = function (?float $p, float $q) {
                        return is_null($p) ? null : (float)($p * $q);
                    };

                    return [
                        $itemNo,
                        $r['Description'] ?? '',
                        $r['Location'] ?? '',
                        (int) ($r['Stock_For'] ?? 0),
                        (int) ($r['Lead Time'] ?? 0),
                        (float) ($r['Consumption'] ?? 0),

                        $sMin,
                        $tMin,
                        $aMin,
                        $fMin,

                        $sMax,
                        $tMax,
                        $aMax,
                        $fMax,

                        $priceBase,              // Price / Base UoM

                        $cost($priceBase, $sMin),
                        $cost($priceBase, $tMin),
                        $cost($priceBase, $aMin),
                        $cost($priceBase, $fMin),

                        $cost($priceBase, $sMax),
                        $cost($priceBase, $tMax),
                        $cost($priceBase, $aMax),
                        $cost($priceBase, $fMax),

                        $baseUom,
                        $priceUom,
                    ];
                }

                public function columnFormats(): array
                {
                    return [
                        'F'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, 
                        'G'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, 
                        'H'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Temp Min
                        'I'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Actual Min
                        'J'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Forecast Min
                        'K'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Suggest Max
                        'L'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Temp Max
                        'M'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Actual Max
                        'N'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Forecast Max

                        'O'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Price / Base UoM

                        'P'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Suggest Min)
                        'Q'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Temp Min)
                        'R'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Actual Min)
                        'S'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Forecast Min)
                        'T'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Suggest Max)
                        'U'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Temp Max)
                        'V'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Actual Max)
                        'W'  => \PhpOffice\PhpSpreadsheet\Style\NumberFormat::FORMAT_NUMBER_00, // Cost (Forecast Max)
                        // X (Base UoM) & Y (Price UoM) are text
                    ];
                }

                public function styles(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet $sheet)
                {
                    $sheet->getStyle('A1:Y1')->getFont()->setBold(true);
                    return [];
                }

                public function registerEvents(): array
                {
                    return [
                        \Maatwebsite\Excel\Events\AfterSheet::class => function ($event) {
                            $event->sheet->getDelegate()->freezePane('A2'); // freeze header
                        },
                    ];
                }
            },
            $fileName
        );
    }


    public function exportMissingVendor(Request $request)
    {
        $start = microtime(true);
        $user = session('user');
        $userName = $user['email'] ?? null;
        $items = session('po_items', []);
        $locationCode = $request->query('location_code');
        $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
        $baseApi = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})";
        $headers = [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000'
            ];
        $step1 = microtime(true);
       

        $today = now()->toDateString();
        $items = collect($items); 
        if ($locationCode) {
            $items = $items->filter(function ($item) use ($locationCode) {
                return isset($item['location_code']) && $item['location_code'] === $locationCode;
            });
        }
         $missingVendorItems = collect($items)->filter(function ($item) {
            return empty($item['vendor_no']) || $item['vendor_no'] == 0;
        });
        $itemNos = $items->pluck('item_no')->unique()->filter()->values();
        $priceList = [];

        $step2 = microtime(true);   
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000'
            ]
        ];

        foreach ($itemNos->chunk(200) as $chunk) {
            $assetFilter = $chunk->map(fn($no) => "Asset_No eq '$no'")->implode(' or ');
            $filter = "SourceNo ne '' and StartingDate le $today and EndingDate lt $today and ($assetFilter)";
            $encodedFilter = urlencode($filter);

            $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
                "/ODataV4/Company('" . $this->companyId . "')/Price_List_Lines?\$filter={$encodedFilter}";

            try {
                $response = $this->client->get($url, $headers);
                $data = json_decode((string)$response->getBody(), true);
                $priceList = array_merge($priceList, $data['value'] ?? []);
            } catch (\Throwable $e) {
                \Log::error("Failed to fetch price list chunk: " . $e->getMessage());
            }
        }

        
        
        $vendorNameMap = session('vendor_map', []);
        
        $inactiveVendorsByItem = collect($priceList)->groupBy('Asset_No');
        
         return Excel::download(new class($missingVendorItems, $inactiveVendorsByItem, $vendorNameMap) implements FromCollection, WithHeadings {
            private $items, $vendorMap, $vendorNameMap;

            public function __construct($items, $vendorMap, $vendorNameMap)
            {
                $this->items = $items;
                $this->vendorMap = $vendorMap;
                $this->vendorNameMap = $vendorNameMap;
            }

            public function collection()
            {
                return $this->items->map(function ($item) {
                    $itemNo = $item['item_no'] ?? '';
                    $vendorInfo = $this->vendorMap[$itemNo][0] ?? null;
                    $vendorNo = $vendorInfo['AssignToNo'] ?? '';
                    $vendorName = $this->vendorNameMap[$vendorNo] ?? 'N/A';

                    return [
                        'Kode'     => $itemNo,
                        'Item' => $item['description'] ?? '',
                        'Supplier' => $vendorName,
                        'Harga Non PPN'   => $vendorInfo['DirectUnitCost'] ?? 'N/A',
                        'UoM'   => $vendorInfo['Unit_of_Measure_Code'] ?? 'N/A'
                    ];
                });
            }

            public function headings(): array
            {
                return ['Kode', 'Item', 'Supplier', 'Harga Non PPN', 'UoM'];
            }
        }, 'missing-vendors.xlsx');
    }

    public function exportPoLines(Request $request)
    {
        $items = collect(session('po_items', []));
        $locationCode = $request->query('location_code');

        $rawStatus = $request->input('status', $request->input('statuses', 'Follow Up PO'));
        if (is_string($rawStatus)) {
            $statusList = array_filter(array_map('trim', explode(',', $rawStatus)));
        } elseif (is_array($rawStatus)) {
            $statusList = array_map('trim', $rawStatus);
        } else {
            $statusList = ['Follow Up PO'];
        }
        $statusSet = collect($statusList)->map(fn ($s) => mb_strtolower($s))->flip(); // O(1) lookup

        if ($locationCode) {
            $items = $items->filter(fn ($it) =>
                isset($it['location_code']) && $it['location_code'] === $locationCode
            );
        }

        $items = $items
        ->filter(fn ($it) => !empty($it['po_lines']) && is_array($it['po_lines']))
        ->filter(function ($it) use ($statusSet) {
            $itemStatus = mb_strtolower(trim($it['status'] ?? $it['item_status'] ?? ''));
            return $itemStatus !== '' && $statusSet->has($itemStatus);
        })
        ->values();

        return \Maatwebsite\Excel\Facades\Excel::download(
            new class($items) implements \Maatwebsite\Excel\Concerns\FromCollection,
                                    \Maatwebsite\Excel\Concerns\WithHeadings,
                                    \Maatwebsite\Excel\Concerns\ShouldAutoSize {
                private $items;
                public function __construct($items) { $this->items = $items; }

                public function headings(): array
                {
                    return [
                        'Item No',
                        'Item Description',
                        'Location',
                        'Document No',
                        'Approved Date',
                        'Vendor',
                        'Status',
                        'Outstanding Qty',
                    ];
                }

                public function collection()
                {
                    $rows = collect();

                    foreach ($this->items as $item) {
                        $itemNo = $item['item_no'] ?? '';
                        $desc   = $item['description'] ?? '';
                        $loc    = $item['location_code'] ?? '';

                        foreach ($item['po_lines'] as $po) {
                            $rows->push([
                                $itemNo,
                                $desc,
                                $loc,
                                $po['document_no'] ?? '',
                                ($po['approved_date'] ?? 'No Date'),
                                $po['vendor'] ?? '',
                                $po['status'] ?? '',
                                $po['outstanding_qty'] ?? 0,
                            ]);
                        }
                    }

                    return $rows;
                }
            },
            'po-lines.xlsx'
        );
    }

    public function exportPoLinesVisible(Request $request)
    {
        $items = collect(session('po_items', []));
        $vendorOptionsByItem = session('vendor_options_by_item', []);
        $vendorNameMap       = session('vendor_map', []);
    
        $selectedLocation = $request->query('location_code');
        $selectedVendors     = (array) $request->input('selected_vendors', []);
        $vendorKeywordsInput = trim((string) $request->query('vendor_keywords', ''));
        $vendorKeywords = collect(explode('|', $vendorKeywordsInput))
                            ->map('trim')->filter()->map(fn($v) => mb_strtolower($v))->values()->all();
    
        $itemKeywordsInput = trim((string) $request->query('item_keywords', ''));
        $itemKeywords = collect(explode('|', $itemKeywordsInput))
                            ->map('trim')->filter()->map(fn($v) => mb_strtolower($v))->values()->all();
    
        $selectedStatus      = (array) $request->input('selected_status', []);
        $selectedStatusItem  = (array) $request->input('selected_status_item', []);
        $selectedDocumentNo  = $request->query('document_no');
        $shipFrom            = $request->query('ship_from');
        $shipTo              = $request->query('ship_to');
    
        $activeParam   = $request->query('active', null);
        $selectedActive = is_null($activeParam) ? null
                        : filter_var($activeParam, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $needShipmentParam = $request->query('need_shipment', null);
        $selectedNeedShipment = is_null($needShipmentParam)
            ? null
            : filter_var($needShipmentParam, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);

    
        $requestedColumns = (array) $request->query('columns', []);
    
        $columnMap = [
            'item_no'        => ['Item No',        fn($i) => $i['item_no'] ?? ''],
            'description'    => ['Item Description', fn($i) => $i['description'] ?? ''],
            'location_code'  => ['Location',       fn($i) => $i['location_code'] ?? ''],
            'min_qty'        => ['min_qty',        fn($i) => $i['min_qty'] ?? ''],
            'max_qty'        => ['max_qty',        fn($i) => $i['max_qty'] ?? ''],
            'stock'          => ['stock',          fn($i) => $i['stock'] ?? ''],
            'on_po'          => ['on_po',          fn($i) => $i['on_po'] ?? ''],
            'in_transfer'    => ['in_transfer',    fn($i) => $i['in_transfer'] ?? ''],
            'qty_to_order'   => ['qty_to_order',   fn($i) => $i['qty_to_order'] ?? ''],
            'status'         => ['Item Status',    fn($i) => $i['status'] ?? ''],
            'Status'         => ['Status (Item)',  fn($i) => $i['Status'] ?? ''],
            'Active'         => ['Active',         fn($i) => isset($i['Active']) ? (is_bool($i['Active']) ? ($i['Active'] ? 'true' : 'false') : (string)$i['Active']) : ''],
            'vendor_no'      => ['Vendor No',      fn($i) => $i['vendor_no'] ?? ''],
            'vendor_name'    => ['Vendor Name',    fn($i) => function() use ($i, $vendorOptionsByItem, $vendorNameMap) {
                                                      $itemNo = $i['item_no'] ?? '';
                                                      $vno    = $i['vendor_no'] ?? '';
                                                      $opt    = $vendorOptionsByItem[$itemNo][$vno]['name'] ?? ($vendorNameMap[$vno] ?? '');
                                                      return $opt ?: '';
                                                  }],
            'unit_cost'      => ['Unit Cost',      fn($i) => function() use ($i, $vendorOptionsByItem) {
                                                      $itemNo = $i['item_no'] ?? '';
                                                      $vno    = $i['vendor_no'] ?? '';
                                                      return $vendorOptionsByItem[$itemNo][$vno]['unit_cost'] ?? '';
                                                  }],
            'transfer_details' => ['Transfer Details', fn($i) => implode(' | ',
                array_map(fn($t) => ($t['document_no'] ?? '').' '.($t['shipment_date'] ?? '').' Qty:'.($t['quantity'] ?? ''),
                    (array)($i['transfer_lines'] ?? [])
                )
            )],
            'po_details' => ['PO details', fn($i) => implode(' | ',
                array_map(fn($p) => ($p['document_no'] ?? '').' '.($p['approved_date'] ?? 'No Date').' '.($p['status'] ?? '').' Qty:'.($p['outstanding_qty'] ?? 0),
                    (array)($i['po_lines'] ?? [])
                )
            )],
            'open_po_details' => ['Open PO details', fn($i) => implode(' | ',
                array_map(fn($p) => ($p['document_no'] ?? '').' '.($p['approved_date'] ?? 'No Date').' '.($p['status'] ?? '').' Qty:'.($p['outstanding_qty'] ?? 0),
                    (array)($i['openPo'] ?? [])
                )
            )],
        ];
    
        $defaultOrder = [
            'item_no','description','location_code','min_qty','max_qty','stock','on_po','in_transfer','qty_to_order','status','Status'
        ];
    
        $columns = array_values(array_filter(
            $requestedColumns ?: $defaultOrder,
            fn($k) => array_key_exists($k, $columnMap)
        ));
    

        $filtered = $items->filter(function ($item) use (
            $selectedLocation, $selectedVendors, $vendorKeywords, $itemKeywords,
            $selectedStatus, $selectedStatusItem, $selectedDocumentNo, $shipFrom,$shipTo, $selectedActive, $vendorOptionsByItem, $vendorNameMap,$selectedNeedShipment
        ) {
            $desc = strtoupper(trim($item['description'] ?? ''));
            if (str_contains($desc, 'FRUIT & VEGETABLE') ||
                str_contains($desc, 'MEAT -') ||
                str_contains($desc, 'SEAFOOD -')) {
                return false;
            }
            if ($selectedLocation && ($item['location_code'] ?? null) !== $selectedLocation) return false;
    
            $itemNo = trim((string)($item['item_no'] ?? ''));
            $itemVendorsMap = $vendorOptionsByItem[$itemNo] ?? [];
            $itemVendorNos  = array_keys($itemVendorsMap);
    
            if (!empty($selectedVendors)) {
                $hasVendor = count(array_intersect($selectedVendors, $itemVendorNos)) > 0
                          || in_array($item['vendor_no'] ?? '', $selectedVendors, true);
                if (!$hasVendor) return false;
            }
    
            if (!empty($vendorKeywords)) {
                $allVendorNamesLower = [];
                if (!empty($itemVendorsMap)) {
                    foreach ($itemVendorsMap as $vno => $vi) {
                        $name = is_array($vi) && isset($vi['name']) ? $vi['name'] : ($vendorNameMap[$vno] ?? null);
                        if ($name) $allVendorNamesLower[] = mb_strtolower($name);
                    }
                } elseif (!empty($item['vendor_no'])) {
                    $name = $vendorNameMap[$item['vendor_no']] ?? null;
                    if ($name) $allVendorNamesLower[] = mb_strtolower($name); // <- fixed typo
                }
                if (!empty($allVendorNamesLower)) {
                    $matched = false;
                    foreach ($vendorKeywords as $kw) {
                        foreach ($allVendorNamesLower as $nm) {
                            if (str_contains($nm, $kw)) { $matched = true; break 2; }
                        }
                    }
                    if (!$matched) return false;
                } else {
                    return false;
                }
            }
    
            if (!empty($itemKeywords)) {
                $searchText = mb_strtolower(($item['item_no'] ?? '') . ' ' . ($item['description'] ?? ''));
                $found = false;
                foreach ($itemKeywords as $kw) {
                    if ($kw !== '' && str_contains($searchText, $kw)) { $found = true; break; }
                }
                if (!$found) return false;
            }
    
            if (!empty($selectedStatus) && !in_array($item['status'] ?? null, $selectedStatus, true)) return false;
            if (!empty($selectedStatusItem) && !in_array($item['Status'] ?? null, $selectedStatusItem, true)) return false;
    
            if ($selectedDocumentNo) {
                $lines = $item['transfer_lines'] ?? [];
                $hasDoc = false;
                foreach ($lines as $line) {
                    if (($line['document_no'] ?? null) === $selectedDocumentNo) { $hasDoc = true; break; }
                }
                if (!$hasDoc) return false;
            }
    
            if ($shipFrom && $shipTo) {
                $lines = $item['transfer_lines'] ?? [];
                $inRange = false;
                foreach ($lines as $line) {
                    $sd = $line['shipment_date'] ?? null;
                    if ($sd && $sd >= $shipFrom && $sd <= $shipTo) { $inRange = true; break; }
                }
                if (!$inRange) return false;
            }
            
            if (!is_null($selectedNeedShipment)) {
                $needVal = (bool)($item['need_shipment'] ?? false);
                if ($needVal !== (bool)$selectedNeedShipment) return false;
            }
    
            if (!is_null($selectedActive)) {
                $activeValue = $item['Active'] ?? null;
                if (is_string($activeValue)) {
                    $activeValue = filter_var($activeValue, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
                }
                if ((bool)$activeValue !== (bool)$selectedActive) return false;
            }
    
            return true;
        })->values();

        $labels = array_map(fn($k) => $columnMap[$k][0], $columns);
    
       return Excel::download(
            new class($filtered, $columns, $columnMap)
                implements \Maatwebsite\Excel\Concerns\FromCollection,
                           \Maatwebsite\Excel\Concerns\WithHeadings,
                           \Maatwebsite\Excel\Concerns\WithEvents
            {
                private $items; private $cols; private $map;
                public function __construct($items, $cols, $map){ $this->items=$items; $this->cols=$cols; $this->map=$map; }
        
                public function headings(): array {
                    return array_map(fn($k) => $this->map[$k][0], $this->cols);
                }
        
                public function collection() {
                    return collect($this->items)->map(function($i){
                        $row = [];
                        foreach ($this->cols as $k) {
                            $val = $this->map[$k][1]($i);
                            if ($val instanceof \Closure) { $val = $val(); }
                            $row[] = is_bool($val) ? ($val ? 'true' : 'false') : $val;
                        }
                        return $row;
                    });
                }
        
                public function registerEvents(): array
                {
                    // widths per DATA KEY (not per letter) so it works with any column order
                    $widthMap = [
                        'description'       => 35, // Item Description
                        'status'            => 18, // Item Status
                        'transfer_details'  => 32, // Transfer Details
                        'po_details'        => 32, // PO details
                        'open_po_details'   => 32, // Open PO details
                        // other columns will get default width below
                    ];
        
                    // bring selected keys into the closure
                    $cols = $this->cols;
        
                    return [
                        \Maatwebsite\Excel\Events\AfterSheet::class => function(\Maatwebsite\Excel\Events\AfterSheet $event) use ($cols, $widthMap) {
                            $sheet = $event->sheet->getDelegate();
        
                            // full used range
                            $lastCol = $sheet->getHighestColumn();
                            $lastRow = $sheet->getHighestRow();
                            $range   = "A1:{$lastCol}{$lastRow}";
        
                            // borders so they show on print
                            $sheet->getStyle($range)->applyFromArray([
                                'borders' => [
                                    'allBorders' => [
                                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                                        'color'       => ['argb' => 'FF000000'],
                                    ],
                                ],
                            ]);
        
                            // header style
                            $sheet->getStyle("A1:{$lastCol}1")->applyFromArray([
                                'font'      => ['bold' => true],
                                'alignment' => ['horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER],
                            ]);
        
                            // wrap + vertical top for all
                            $sheet->getStyle($range)->getAlignment()->setWrapText(true);
                            $sheet->getStyle($range)->getAlignment()->setVertical(
                                \PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_TOP
                            );
        
                            // column widths by key (robust to column order)
                            $sheet->getDefaultColumnDimension()->setWidth(10); // baseline
                            foreach ($cols as $idx => $key) {
                                $letter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($idx + 1);
                                // make sure autosize can’t override manual widths
                                $sheet->getColumnDimension($letter)->setAutoSize(false);
                                if (isset($widthMap[$key])) {
                                    $sheet->getColumnDimension($letter)->setWidth($widthMap[$key]);
                                } else {
                                    // narrow numeric-ish columns look nicer around 10–12
                                    $sheet->getColumnDimension($letter)->setWidth(10);
                                }
                            }
        
                            // page setup – A4, portrait, keep 100% scale (no "fit to 1 page" shrinking)
                            $pageSetup = $sheet->getPageSetup();
                            $pageSetup->setPaperSize(\PhpOffice\PhpSpreadsheet\Worksheet\PageSetup::PAPERSIZE_A4);
                            $pageSetup->setOrientation(\PhpOffice\PhpSpreadsheet\Worksheet\PageSetup::ORIENTATION_PORTRAIT);
                            $pageSetup->setScale(100); // << do NOT use setFitToWidth here
                            $sheet->getPageMargins()->setTop(0.4)->setRight(0.3)->setBottom(0.4)->setLeft(0.3);
        
                            // repeat header row on each page
                            $pageSetup->setRowsToRepeatAtTopByStartAndEnd(1, 1);
        
                            // borders are enough; hide gridlines on screen/print
                            $sheet->setShowGridlines(false);
                            $sheet->setPrintGridlines(false);
        
                            // print area
                            $sheet->getPageSetup()->setPrintArea($range);
                        },
                    ];
                }
            },
            'visible-table.xlsx'
        );
    }

    public function patchItemField(Request $request)
    {
        $itemNo = $request->input('item_no');
        $variantCode = $request->input('variant_code', '');
        $location = $request->input('location_code');

        $fieldsToUpdate = [];
        $fieldaToUpdate = [];

        if ($request->has('Status')) {
            $fieldsToUpdate['Status'] = $request->input('Status');
            $fieldaToUpdate['Status'] = $request->input('Status');
        }
        if ($request->has('Comment')) {
            $fieldsToUpdate['Comment'] = $request->input('Comment');
            $fieldaToUpdate['Comment'] = $request->input('Comment');
        }
        if ($request->has('Active')) {
            $fieldsToUpdate['Active'] = filter_var($request->input('Active'), FILTER_VALIDATE_BOOLEAN);
            $fieldaToUpdate['Active'] = filter_var($request->input('Active'), FILTER_VALIDATE_BOOLEAN);
        }
        
        
        try {
            $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
            $company = "/ODataV4/Company('$this->companyId')/APIStockkeeping";
            $patchUrl = $baseUrl . $company .
                "(Item_No='" . rawurlencode($itemNo) . "'," .
                "Variant_Code='" . rawurlencode($variantCode) . "'," .
                "Location_Code='" . rawurlencode($location) . "')";

            $headers = [
                'Authorization' => "Bearer " . $this->getAccessToken(),
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'If-Match' => '*'
            ];

            $response = $this->client->request('PATCH', $patchUrl, [
                'headers' => $headers,
                'body' => json_encode($fieldsToUpdate)
            ]);

            return response()->json(['message' => 'Item updated.']);
        } catch (\Exception $e) {
             \Log::warning("PATCH to APIStockkeeping failed for $itemNo at $location: " . $e->getMessage());
            
            try {
                $fallbackUrl = $baseUrl . "/ODataV4/Company('$this->companyId')/ItemInvenLoc" .
                    "(ItemNo='" . rawurlencode($itemNo) . "'," .
                    "Location='" . rawurlencode($location) . "')";

                $this->client->request('PATCH', $fallbackUrl, [
                    'headers' => $headers,
                    'body'    => json_encode($fieldaToUpdate)
                ]);
                \Log::info("Fallback PATCH succeeded for $itemNo at $location");
                return response()->json("message' => 'Item updated.");
            } catch (\Exception $fallbackError) {
                \Log::error("Both APIStockkeeping and fallback ItemInvenLoc PATCH failed for $itemNo at $location: " . $fallbackError->getMessage());
                return response()->json(['error' => 'Failed to update item.'], 500);
            }
            return response()->json(['error' => 'Failed to update item.'], 500);
        }
    }



    public function create(Request $request, BusinessCentralService $bcService)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;
        $selectedItems = array_unique($request->input('selected_items', []));
        $vendorMap = $request->input('vendor_no', []);
        $qtyMap = $request->input('qty_to_order', []);
        $unitCost = $request->input('unit_cost', []);
        $locationMap = $request->input('location_code', []);

        $groupedByVendor = [];
        
        foreach ($selectedItems as $itemNo) {
            $vendor = $vendorMap[$itemNo] ?? null;
            $qty = $qtyMap[$itemNo] ?? 0;
            $cost = $unitCost[$itemNo];
            $loc  = $locationMap[$itemNo] ?? '';

            if (!$vendor || $vendor === '0' || $qty <= 0) continue;

            $groupedByVendor[$vendor][] = [
                'item_no' => $itemNo,
                'quantity' => $qty,
                'cost' => $cost,
                'location' => $loc
            ];
        }
        $results = [];
        $successes = [];
        $failures  = [];

        foreach ($groupedByVendor as $vendorNo => $items) {
             $poLocation = $items[0]['location'] ?? '';
             $po = $bcService->createPurchaseOrderForVendor($vendorNo, $userName, $poLocation);
             $promises = [];

             foreach ($items as $line) {
                $response = $bcService->addPurchaseLineToPO(
                    $po['number'],
                    $line['item_no'],
                    $line['quantity'],
                    $line['cost'],
                    $line['location'],
                    $userName,
                    $po['id']
                );
            }

             $results[] = [
                 'vendor'     => $vendorNo,
                 'po_no'      => $po['number'] ?? 'N/A',
                 'line_count' => count($items),
             ];
         }
    
        return response()->json([
             'success' => 'Purchase Orders created successfully.',
             'createdPOs' => $results,
        ]);
    }

    public function updateSku(Request $request)
    {
        set_time_limit(3000);
        $updatedRows = $request->input('rows', []);
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $company = "/ODataV4/Company('" . $this->companyId . "')/SKUImport";
        $headers = [
            'Authorization' => "Bearer {$this->token}",
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'If-Match' => '*'
        ];

        foreach ($updatedRows as $row) {
            $patchUrl = $baseUrl . $company . "(Item_No='" . rawurlencode($row['Item_No']) . "',Location='" . rawurlencode($row['Location']) . "')";

            $body = [
                'StockFor' => $row['Stock_For'],
                'LeadTime' => $row['Lead Time'],
                'MinInven' => $row['ActMinQty'],
                'MaxInven' => $row['ActMaxQty'],
                'Comment' => $row['Comment'] ?? ''
            ];

            try {
                $this->client->request('PATCH', $patchUrl, [
                    'headers' => $headers,
                    'body' => json_encode($body)
                ]);
            } catch (\Exception $e) {
                \Log::warning("PATCH failed for SKU {$row['Item_No']}, trying POST. Reason: " . $e->getMessage());

                try {
                    $this->client->request('POST', $baseUrl . $company, [
                        'headers' => $headers,
                        'body' => json_encode(array_merge([
                            'Item_No' => $row['Item_No'],
                            'Location' => $row['Location']
                        ], $body))
                    ]);
                } catch (\Exception $e2) {
                    \Log::error("Failed to POST SKU {$row['Item_No']}: " . $e2->getMessage());
                }
            }
        }

        return response()->json(['message' => 'SKU update process completed']);
    }


    public function updateStockkeeping(Request $request)
    {
        set_time_limit(6000);

        $selectedRows = $request->input('rows', []);

        $this->updateSku($request);

        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $company = "/ODataV4/Company('" . $this->companyId . "')/APIStockkeeping";

        $headers = [
            'Authorization' => "Bearer {$this->token}", 
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'If-Match' => '*'
        ];

        foreach ($selectedRows as $row) {
            try {
                $patchUrl = $baseUrl . $company .
                    "(Item_No='" . rawurlencode($row['Item_No']) . "'," .
                    "Variant_Code='" . rawurlencode($row['Variant_Code'] ?? '') . "'," .
                    "Location_Code='" . rawurlencode($row['Location']) . "')";

                $body = [
                    'MinInven' => $row['ActMinQty'],
                    'MaxInven' => $row['ActMaxQty']
                ];

                $this->client->request('PATCH', $patchUrl, [
                    'headers' => $headers,
                    'body'    => json_encode($body)
                ]);
            } catch (\Exception $e) {
                \Log::error("Failed to patch Stockkeeping for {$row['Item_No']} - {$row['Location']}: " . $e->getMessage());
                return response()->json(['error' => 'Stockkeeping update failed. See logs.'], 500);
            }
        }

        return response()->json(['message' => 'Stockkeeping updated successfully']);
    }




}
