<?php

namespace App\Http\Controllers;

use App\Models\UploadedForm;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class FormController extends Controller
{
    public function uploadForm(Request $request)
    {
        $user = auth()->user();

        $query = UploadedForm::with('user')
            ->where('status', 0); // hanya yang belum sign

        // ======================
        // ROLE-BASED VISIBILITY
        // ======================
        if (
            $user->isAdmin()
            ||
            str_contains(strtoupper($user->name), 'ADMIN')
        ) {
        }
        elseif ($user->isApprover() ) {
            if (str_contains(strtoupper($user->name), 'TEGUH') ) {
                $query->where('flow', 'ENGINEERING');
            }
            elseif (str_contains(strtoupper($user->name), 'MICHAEL DANIEL')) {
                $query->where('flow', 'BUILDING');
            }
            else {
                $query->whereRaw('1=0');
            }
        }
        else {
            // USER → hanya file miliknya
            //$query->where('user_id', $user->id);
        }

        // ======================
        // SEARCH
        // ======================
        if ($request->filled('q')) {
            $q = $request->q;

            $query->where(function ($sub) use ($q) {
                $sub->where('title', 'like', "%$q%")
                    ->orWhere('original_name', 'like', "%$q%")
                    ->orWhereHas('user', fn ($u) =>
                        $u->where('name', 'like', "%$q%")
                    );
            });
        }

        // ======================
        // SORTING
        // ======================
        $sort = $request->get('sort', 'created_at');
        $dir  = $request->get('dir', 'desc');

        $forms = $query->orderBy($sort, $dir)
                    ->paginate(10)
                    ->withQueryString();

        return view('forms-upload', compact('forms'));
    }


    public function storePdf(Request $request)
    {
        $request->validate([
            'pdf'   => 'required|array',
            'pdf.*' => 'file|mimes:pdf|max:5120',
            'flow'  => 'required|in:BUILDING,ENGINEERING',
        ]);

        foreach ($request->file('pdf') as $file) {
            $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);

            $path = $file->store('forms', 'public');

            UploadedForm::create([
                'title'         => $originalName,
                'original_name' => $file->getClientOriginalName(),
                'path'          => $path,
                'size'          => $file->getSize(),
                'status'        => '0',
                'user_id'       => auth()->id(),
                'flow'          => $request->flow,
            ]);
        }

        return redirect()->back()->with('success', 'PDFs uploaded successfully');
    }

    public function deletePdf(UploadedForm $form)
    {
        // Only admin or the uploader can delete
        if (auth()->user()->id !== $form->user_id && !auth()->user()->isAdmin()) {
            abort(403, 'You are not allowed to delete this form.');
        }

        // Delete the file from storage
        if (Storage::disk('public')->exists($form->path)) {
            Storage::disk('public')->delete($form->path);
        }

        // Delete the database record
        $form->delete();

        return redirect()->back()->with('success', 'PDF form deleted successfully.');
    }

    public function batchSign(Request $request)
    {
        $user = auth()->user();

        if (!$user || (!$user->isApprover() && !$user->isAdmin())) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $ids = collect(explode(',', (string) $request->input('ids')))
        ->map(fn ($id) => (int) trim($id))
        ->filter()
        ->values()
        ->toArray();

        if (!is_array($ids) || empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No documents selected']);
        }
        $signed = 0;
        $forms = UploadedForm::whereIn('id', $ids)
            ->where('status', '0')
            ->get();

        foreach ($forms as $form) {
            $boolea = $this->canUserSignForm($user, $form);
            // Log::debug($boolea);
            // if (!$this->canUserSignForm($user, $form)) {
            //     continue; 
            // }

            $this->signPdf($form);

            $signed++;
        }

        return back()->with('success', "$signed document(s) signed successfully.");
    }


    public function signPdf(UploadedForm $form)
    {
        $user = auth()->user();
    
        if (!$user->isApprover() && !$user->isAdmin()) {
            abort(403, 'Only approvers can sign PDFs');
        }
    
        // =============================
        // DETERMINE SIGNER
        // =============================
        if (str_contains(strtoupper($user->name), 'TEGUH') ) {
            $signerName = 'Teguh Sugiarto';
            $signatureFile = 'Teguh Sugiarto.png';
        } elseif (str_contains(strtoupper($user->name), 'MICHAEL DANIEL') || $user->isAdmin()) {
            $signerName = 'Michael Daniel';
            $signatureFile = 'Michael Daniel.png';
        } else {
            return back()->with('error', 'Signature not configured for this user.');
        }
    
        // =============================
        // FILE PATHS
        // =============================
        $inputPath = storage_path('app/public/' . $form->path);
        $sigPath   = storage_path('app/public/signature/' . $signatureFile);
        $response = Http::timeout(120)
            ->attach(
                'pdf',
                file_get_contents($inputPath),
                $form->original_name
            )
            ->attach(
                'signature',
                file_get_contents($sigPath),
                $signatureFile
            )
            ->post('https://pawindo.site/sign/', [
                'signer' => $signerName,
            ]);

        if (!$response->successful()) {
            return back()->with(
                'error',
                'Signing failed: ' . $response->body()
            );
        }
    
        // =============================
        // SAVE SIGNED PDF
        // =============================
        $signedPath = 'forms/signed-' . $form->original_name;
    
        Storage::disk('public')->put(
            $signedPath,
            $response->body()
        );
    
        // =============================
        // UPDATE DB
        // =============================
        $form->update([
            'path'      => $signedPath,
            'status'    => '1',
            'signed_by' => $user->id,
        ]);
    
        return back()->with('success', 'PDF signed successfully!');
    }


    private function canUserSignForm($user, $form): bool
    {
        if ($user->isAdmin()) return true;

        if ($form->flow === 'ENGINEERING' &&
            str_contains(strtoupper($user->name), 'TEGUH')) {
            return true;
        }

        if ($form->flow === 'BUILDING' &&
            str_contains(strtoupper($user->name), 'MICHAEL DANIEL')) {
            return true;
        }

        return false;
    }

    public function indexSignedForms()
    {
        // Get all signed forms
        $forms = UploadedForm::where('status', 1)
            ->orderBy('created_at', 'desc')
            ->get();

        return view('signed-forms', compact('forms'));
    }
}
