@extends('layouts.app')

@section('title', 'Upload PDF Forms')

@section('content')


@if(session('success'))
    <div class="alert alert-success py-2">
        {{ session('success') }}
    </div>
@endif

@if(session('error'))
    <div class="alert alert-danger py-2">
        {{ session('error') }}
    </div>
@endif

<!-- Page Header -->
<div class="mb-4">
    <h3 class="fw-bold mb-1">📄 Upload PDF Forms</h3>
    <p class="text-muted mb-0">
        Upload and manage Payment Request Forms
    </p>
</div>
    

<div class="row g-4">
    <div class="col-lg-5">
        <div class="card card-soft">
            <div class="card-body">

                <form method="POST" action="{{ route('forms.upload') }}" enctype="multipart/form-data">
                    @csrf
                    <div class="mb-3">
                        <label class="form-label fw-semibold">Flow</label>
                        <select name="flow" class="form-select" required>
                            <option value="">-- Select Flow --</option>
                            <option value="BUILDING">BUILDING</option>
                            <option value="ENGINEERING">ENGINEERING</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label fw-semibold">Upload PDF Forms</label>
                        <input
                            type="file"
                            name="pdf[]"
                            class="form-control"
                            accept="application/pdf"
                            multiple
                            required
                        >
                        <small class="text-muted">You can select multiple PDF files</small>
                    </div>

                    @error('pdf.*')
                        <div class="text-danger">{{ $message }}</div>
                    @enderror

                    <button class="btn btn-primary">
                        <i class="fa-solid fa-upload me-1"></i> Upload
                    </button>
                </form>
            </div>
        </div>
    </div>

    <div class="col-lg-7">
        <div class="card card-soft h-100">
            <div class="card-body">

                <h6 class="fw-semibold mb-3">Unsigned Forms</h6>
                <div class="row g-2 mb-3 align-items-center">
                    <div class="col">
                        <input type="text" id="searchInput"
                            class="form-control"
                            placeholder="Search file / uploader">
                    </div>

                    <div class="col-auto">
                        <select id="sortBy" class="form-select">
                            <option value="created">Created Date</option>
                            <option value="title">Title</option>
                            <option value="flow">Flow</option>
                        </select>
                    </div>

                    <div class="col-auto">
                        <select id="sortDir" class="form-select">
                            <option value="desc">Desc</option>
                            <option value="asc">Asc</option>
                        </select>
                    </div>
                </div>
                @if(session('user.role') === 'ADMIN' || session('user.role') === 'APPROVER')
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <div>
                            <button id="selectAllBtn" class="btn btn-sm btn-outline-secondary">
                                Select All
                            </button>
                            <button id="clearAllBtn" class="btn btn-sm btn-outline-secondary">
                                Clear
                            </button>
                        </div>
                        <form id="batchSignForm" method="POST" action="{{ route('forms.batchSign') }}">
                            @csrf
                            <input type="hidden" name="ids" id="batchIds">
                        </form>
                        <button id="batchSignBtn"
                                type="button"
                                class="btn btn-sm btn-success"
                                disabled>
                            <i class="fa-solid fa-pen-fancy me-1"></i>
                            Sign Selected
                        </button>
                    </div>
                    @endif

                @if($forms->isEmpty())
                    <div class="text-muted small">
                        No PDF forms pending signature.
                    </div>
                @else
                    <div class="row g-3">
                        @foreach($forms as $form)
                            @php
                                $canSign = false;

                                if(session('user.role') === 'ADMIN') {
                                    $canSign = true;
                                } 
                                elseif($form->flow === 'ENGINEERING' && str_contains(strtoupper(session('user.name')), 'TEGUH')) {
                                    $canSign = true;
                                } 
                                elseif($form->flow === 'BUILDING' && str_contains(strtoupper(session('user.name')), 'MICHAEL DANIEL')) {
                                    $canSign = true;
                                }
                            @endphp
                            
                            <div class="col-12">
                                
                                <div class="card card-hover p-3 d-flex flex-row align-items-center justify-content-between shadow-sm form-item"
                                    data-title="{{ strtolower($form->title ?? $form->original_name) }}"
                                    data-user="{{ strtolower($form->user->name ?? '') }}"
                                    data-flow="{{ strtolower($form->flow) }}"
                                    data-created="{{ $form->created_at->timestamp }}">

                                    <!-- File Info -->
                                    <div class="d-flex align-items-center gap-3">
                                        <div class="pdf-icon">
                                            <i class="fa-solid fa-file-pdf fa-lg"></i>
                                        </div>
                                        <div>
                                            <div class="fw-semibold">{{ $form->title ?? $form->original_name }}</div>
                                            <small class="text-muted">
                                                {{ number_format($form->size / 1024, 1) }} KB
                                            </small>

                                            <div class="text-muted small mt-1">
                                                Uploaded by: {{ $form->user->name ?? 'Unknown' }}
                                            </div>

                                            <div class="text-muted small">
                                                Created: {{ $form->created_at->format('d M Y, H:i') }} <br>
                                                Updated: {{ $form->updated_at->eq($form->created_at) ? '-' : $form->updated_at->format('d M Y, H:i') }}
                                            </div>

                                            <div class="mt-1">
                                                <span class="badge bg-warning text-dark">Not Signed</span> <span class="badge bg-warning text-dark ml-2">{{ $form->flow }}</span>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="d-flex gap-2">
                                        @if($canSign && (int)$form->status === 0)
                                            <input type="checkbox"
                                                class="form-check-input batch-checkbox me-2"
                                                value="{{ $form->id }}">
                                        @endif
                                        <a href="{{ asset('public/storage/'.$form->path) }}"
                                        target="_blank"
                                        class="btn btn-outline-primary btn-sm">
                                            <i class="fa-solid fa-eye me-1"></i> View
                                        </a>
                                        @if($canSign && $form->status === '0')
                                            <form method="POST" action="{{ route('forms.sign', $form->id) }}">
                                                @csrf
                                                <button class="btn btn-sm btn-success">
                                                    <i class="fa-solid fa-pen-fancy me-1"></i> Sign
                                                </button>
                                            </form>
                                        @endif

                                        @if(session('user.role') === 'ADMIN' || session('user.id') === $form->user_id)
                                            <form method="POST" action="{{ route('forms.delete', $form->id) }}" onsubmit="return confirm('Are you sure you want to delete this form?');">
                                                @csrf
                                                @method('DELETE')
                                                <button class="btn btn-sm btn-outline-danger">
                                                    <i class="fa-solid fa-trash me-1"></i> Delete
                                                </button>
                                            </form>
                                        @endif
                                    </div>

                                </div>
                            </div>
                        @endforeach
                    </div>
                @endif

            </div>
        </div>
    </div>
@endsection
@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', () => {

    const searchInput = document.getElementById('searchInput');
    const sortBy = document.getElementById('sortBy');
    const sortDir = document.getElementById('sortDir');
    const container = document.querySelector('.row.g-3');

    const batchBtn   = document.getElementById('batchSignBtn');
    const batchForm  = document.getElementById('batchSignForm');
    const batchIds   = document.getElementById('batchIds');
    const selectAllBtn = document.getElementById('selectAllBtn');
    const clearAllBtn  = document.getElementById('clearAllBtn');

    const checkboxes = () =>
        Array.from(document.querySelectorAll('.batch-checkbox'));

    let items = Array.from(container.querySelectorAll('.col-12'));

    /* =========================
       SEARCH + SORT
    ========================= */
    function filterAndSort() {
        const q = searchInput.value.toLowerCase();
        const sortKey = sortBy.value;
        const direction = sortDir.value;

        items.forEach(item => {
            const card = item.querySelector('.form-item');
            const match =
                card.dataset.title.includes(q) ||
                card.dataset.user.includes(q) ||
                card.dataset.flow.includes(q);

            item.style.display = match ? '' : 'none';
        });

        items.sort((a, b) => {
            const A = a.querySelector('.form-item');
            const B = b.querySelector('.form-item');

            let va, vb;
            if (sortKey === 'created') {
                va = parseInt(A.dataset.created);
                vb = parseInt(B.dataset.created);
            } else {
                va = A.dataset[sortKey];
                vb = B.dataset[sortKey];
            }

            return direction === 'asc'
                ? va > vb ? 1 : -1
                : va < vb ? 1 : -1;
        });

        items.forEach(item => container.appendChild(item));
    }

    searchInput.addEventListener('input', filterAndSort);
    sortBy.addEventListener('change', filterAndSort);
    sortDir.addEventListener('change', filterAndSort);

    /* =========================
       BATCH SIGN
    ========================= */
    function updateButtonState() {
        batchBtn.disabled = checkboxes().filter(cb => cb.checked).length === 0;
    }

    document.addEventListener('change', e => {
        if (e.target.classList.contains('batch-checkbox')) {
            updateButtonState();
        }
    });

    batchBtn.addEventListener('click', () => {
        const ids = checkboxes()
            .filter(cb => cb.checked)
            .map(cb => cb.value);

        if (!ids.length) return;

        if (!confirm(`Sign ${ids.length} selected document(s)?`)) return;

        batchIds.value = ids.join(',');
        batchForm.submit();
    });

    selectAllBtn?.addEventListener('click', e => {
        e.preventDefault();
        checkboxes().forEach(cb => cb.checked = true);
        updateButtonState();
    });

    clearAllBtn?.addEventListener('click', e => {
        e.preventDefault();
        checkboxes().forEach(cb => cb.checked = false);
        updateButtonState();
    });

});
</script>

@endpush

@push('styles')
<style>
.card-soft {
    border: 0;
    border-radius: 14px;
    box-shadow: 0 8px 24px rgba(0,0,0,.05);
}

.pdf-icon {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    background: rgba(220,38,38,.12);
    color: #dc2626;
    display: flex;
    align-items: center;
    justify-content: center;
}

body[data-theme="dark"] .card-soft {
    background: #0f172a;
}

body[data-theme="dark"] .pdf-icon {
    background: rgba(220,38,38,.25);
}
.card-hover {
    border-radius: 12px;
    transition: transform 0.2s, box-shadow 0.2s;
    background: #ffffff;
}

.card-hover:hover {
    transform: translateY(-3px);
    box-shadow: 0 12px 24px rgba(0,0,0,.1);
}

body[data-theme="dark"] .card-hover {
    background: #1e293b;
}

.badge {
    font-size: 0.75rem;
    padding: 0.35em 0.6em;
    border-radius: 10px;
}

body[data-theme="dark"] .badge.bg-warning.text-dark {
    background-color: #facc15 !important;
    color: #1e293b !important;
}
</style>
@endpush
