<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Laravel\Socialite\Facades\Socialite;
use Carbon\Carbon;

class AzureAuthController extends Controller
{
    protected $authService;

    public function __construct(\App\Services\AzureAuthService $authService)
    {
        $this->authService = $authService;
    }

    /**
     * ============================================================
     * REDIRECT TO MICROSOFT LOGIN
     * ============================================================
     */
    public function redirectToProvider()
    {
        // Jika token masih valid (< 24 jam), langsung masuk
        if (session()->has('graph.expires_at')) {
            $expiresAt = Carbon::parse(session('graph.expires_at'));

            if (now()->lessThan($expiresAt)) {
                return redirect('/pbi-dashboards');
            }
        }

        // Bersihkan session lama agar tidak terjadi state conflict

        return Socialite::driver('microsoft')
            ->scopes(['openid', 'offline_access', 'User.Read', 'Files.Read'])
            ->stateless()
            ->redirect();
    }

    /**
     * ============================================================
     * MICROSOFT CALLBACK HANDLER
     * ============================================================
     */
    public function handleProviderCallback(Request $request)
    {
        try {

            // ================================
            // AMBIL USER DARI MICROSOFT
            // ================================
            $msUser = Socialite::driver('microsoft')
                ->stateless()
                ->user();

            $email = strtolower($msUser->getEmail());
            $name  = $msUser->getName();
            $role  = null;

            // ================================
            // ROLE MAPPING
            // ================================
            if (
                str_contains($email, 'adnyani') ||
                str_contains($email, 'titania')
            ) {
                $role = 'Regent';
            } elseif (
                str_contains($email, 'pebri') ||
                str_contains($email, 'pratiwi') ||
                str_contains($email, 'feybe')
            ) {
                $role = 'HIN';
            } elseif (
                str_contains($email, 'admin') ||
                str_contains($email, 'michael') ||
                str_contains($email, 'kristina') ||
                str_contains($email, 'costingho') ||
                str_contains($email, 'erlina')
            ) {
                $role = 'SUPER';
            } else {
                $role = 'Regent';
            }

            // ================================
            // VALIDASI ROLE
            // ================================
            if (!in_array(strtoupper($role), ['SUPER', 'ADMIN'])) {
                $this->forceLogout($request);
                return redirect('/login');
            }

            // ================================
            // HITUNG EXPIRY TOKEN (MAX 24 JAM)
            // ================================
            $expiresIn = (int) ($msUser->expiresIn ?? 3600);

            $expiresAt = now()->addSeconds(
                min($expiresIn, 86400) // maksimal 24 jam
            );

            // ================================
            // SIMPAN SESSION (TANPA DATABASE)
            // ================================
            session([
                'user' => [
                    'name'  => $name,
                    'email' => $email,
                    'role'  => $role,
                    'login_at' => now()->toDateTimeString(),
                ],
                'graph' => [
                    'access_token'  => $msUser->token,
                    'refresh_token' => $msUser->refreshToken ?? null,
                    'expires_at'    => $expiresAt->toIso8601String(),
                ],
            ]);

            return redirect('/pbi-dashboards');

        } catch (\Throwable $e) {

            Log::error('Azure Login Failed', [
                'message' => $e->getMessage(),
                'file'    => $e->getFile(),
                'line'    => $e->getLine(),
            ]);

            $this->forceLogout($request);
            return redirect('/login');
        }
    }

    /**
     * ============================================================
     * LOGOUT
     * ============================================================
     */
    public function logout(Request $request)
    {
        $this->forceLogout($request);

        $microsoftLogoutUrl =
            'https://login.microsoftonline.com/common/oauth2/v2.0/logout' .
            '?post_logout_redirect_uri=' . urlencode(route('login'));

        return redirect($microsoftLogoutUrl);
    }

    /**
     * ============================================================
     * SESSION CLEANER
     * ============================================================
     */
    protected function forceLogout(Request $request): void
    {
        $request->session()->invalidate();
        $request->session()->flush();
        $request->session()->regenerateToken();
    }
}
