<?php

namespace App\Http\Controllers;

use App\Models\ApprovalHistory;
use App\Models\PoRekap;
use App\Services\BusinessCentralService;
use App\Services\DataSyncService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class ApprovalController extends Controller
{
    private $bcService;
    private $dataSyncService;

    public function __construct(
        BusinessCentralService $bcService,
        DataSyncService $dataSyncService
    ) {
        $this->bcService = $bcService;
        $this->dataSyncService = $dataSyncService;
    }

    public function bulkApprove(Request $request)
{
    try {
        // Validasi
        $validated = $request->validate([
            'document_numbers' => 'required|array|min:1',
            'document_numbers.*' => 'required|string',
            'company' => 'required|string|in:TBI,CI',
            'category' => 'required|string',
            'remarks' => 'nullable|string|max:500',
        ]);

        $user = Auth::user();
        $documentNumbers = $request->document_numbers;
        $company = $request->company;
        $category = $request->category;
        $remarks = $request->remarks;

        $results = [
            'success' => [],
            'failed' => [],
            'skipped' => []
        ];

        foreach ($documentNumbers as $docNo) {
            try {
                // Check if already processed
                $alreadyProcessed = ApprovalHistory::where('document_no', $docNo)
                    ->where('approver_id', $user->approver_id)
                    ->exists();

                if ($alreadyProcessed) {
                    $results['skipped'][] = [
                        'document_no' => $docNo,
                        'reason' => 'Already processed'
                    ];
                    continue;
                }

                \Log::info('Bulk approval request', [
                    'document_no' => $docNo,
                    'company' => $company,
                    'category' => $category,
                    'approver_id' => $user->approver_id
                ]);

                // Verify ownership
                $po = PoRekap::where('document_no', $docNo)
                    ->where('company', $company)
                    ->where('category', $category)
                    ->where('approver_id', $user->approver_id)
                    ->first();

                if (!$po) {
                    $results['failed'][] = [
                        'document_no' => $docNo,
                        'error' => 'PO not found or access denied'
                    ];
                    continue;
                }

                // Call BC API to approve
                $response = $this->bcService->approvePO($company, $docNo, $user->approver_id);

                // Save to history
                ApprovalHistory::create([
                    'document_no' => $docNo,
                    'company' => $company,
                    'category' => $category,
                    'approver_id' => $user->approver_id,
                    'approver_email' => $user->email,
                    'action' => 'approved',
                    'remarks' => $remarks,
                    'action_date' => now(),
                    'response_data' => json_encode($response),
                ]);

                $results['success'][] = $docNo;

                \Log::info('Bulk approval successful', ['document_no' => $docNo]);

            } catch (\Exception $e) {
                \Log::error('Bulk approval item failed', [
                    'document_no' => $docNo,
                    'error' => $e->getMessage(),
                    'line' => $e->getLine()
                ]);

                $results['failed'][] = [
                    'document_no' => $docNo,
                    'error' => $e->getMessage()
                ];
            }
        }

        // Re-sync data after bulk action
        try {
            $this->dataSyncService->resyncAfterAction($company, $category, $user->approver_id);
        } catch (\Exception $e) {
            \Log::error('Re-sync failed', ['error' => $e->getMessage()]);
        }

        $successCount = count($results['success']);
        $failedCount = count($results['failed']);
        $skippedCount = count($results['skipped']);

        $message = "Successfully approved {$successCount} PO(s)";
        if ($skippedCount > 0) {
            $message .= ", {$skippedCount} already processed";
        }
        if ($failedCount > 0) {
            $message .= ", {$failedCount} failed";
        }

        return response()->json([
            'success' => $successCount > 0,
            'message' => $message,
            'results' => $results
        ]);

    } catch (\Illuminate\Validation\ValidationException $e) {
        \Log::error('Validation failed', ['errors' => $e->errors()]);
        
        return response()->json([
            'success' => false,
            'message' => 'Validation failed',
            'errors' => $e->errors()
        ], 422);
        
    } catch (\Exception $e) {
        \Log::error('Bulk approve critical error', [
            'error' => $e->getMessage(),
            'line' => $e->getLine(),
            'file' => $e->getFile()
        ]);

        return response()->json([
            'success' => false,
            'message' => 'Server error: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Bulk Reject Purchase Orders
     */
    public function bulkReject(Request $request)
    {
        $request->validate([
            'document_numbers' => 'required|array|min:1',
            'document_numbers.*' => 'required|string',
            'company' => 'required|string|in:TBI,CI',
            'category' => 'required|string',
            'remarks' => 'nullable|string|max:500',
        ]);

        $user = Auth::user();
        $documentNumbers = $request->document_numbers;
        $company = $request->company;
        $category = $request->category;
        $remarks = $request->remarks;

        $results = [
            'success' => [],
            'failed' => [],
            'skipped' => []
        ];

        foreach ($documentNumbers as $docNo) {
            try {
                // Check if already processed
                $alreadyProcessed = ApprovalHistory::where('document_no', $docNo)
                    ->where('approver_id', $user->approver_id)
                    ->exists();

                if ($alreadyProcessed) {
                    $results['skipped'][] = [
                        'document_no' => $docNo,
                        'reason' => 'Already processed'
                    ];
                    continue;
                }

                Log::info('Bulk rejection request', [
                    'document_no' => $docNo,
                    'company' => $company,
                    'category' => $category,
                    'approver_id' => $user->approver_id
                ]);

                // Verify ownership
                $po = PoRekap::where('document_no', $docNo)
                    ->where('company', $company)
                    ->where('category', $category)
                    ->where('approver_id', $user->approver_id)
                    ->first();

                if (!$po) {
                    $results['failed'][] = [
                        'document_no' => $docNo,
                        'error' => 'PO not found or access denied'
                    ];
                    continue;
                }

                // Call BC API to reject
                $response = $this->bcService->rejectPO($company, $docNo, $user->approver_id);

                // Save to history
                ApprovalHistory::create([
                    'document_no' => $docNo,
                    'company' => $company,
                    'category' => $category,
                    'approver_id' => $user->approver_id,
                    'approver_email' => $user->email,
                    'action' => 'rejected',
                    'remarks' => $remarks,
                    'action_date' => now(),
                    'response_data' => $response,
                ]);

                $results['success'][] = $docNo;

                Log::info('Bulk rejection successful', ['document_no' => $docNo]);

            } catch (\Exception $e) {
                Log::error('Bulk rejection failed', [
                    'document_no' => $docNo,
                    'error' => $e->getMessage()
                ]);

                $results['failed'][] = [
                    'document_no' => $docNo,
                    'error' => $e->getMessage()
                ];
            }
        }

        // Re-sync data after bulk action (to update status in BC)
        try {
            $this->dataSyncService->resyncAfterAction($company, $category, $user->approver_id);
        } catch (\Exception $e) {
            Log::error('Re-sync after bulk rejection failed', ['error' => $e->getMessage()]);
        }

        $successCount = count($results['success']);
        $failedCount = count($results['failed']);
        $skippedCount = count($results['skipped']);

        $message = "Successfully rejected {$successCount} PO(s)";
        if ($skippedCount > 0) {
            $message .= ", {$skippedCount} already processed";
        }
        if ($failedCount > 0) {
            $message .= ", {$failedCount} failed";
        }

        return response()->json([
            'success' => $successCount > 0,
            'message' => $message,
            'results' => $results
        ]);
    }

    /**
     * Single Approve (legacy - keep for backward compatibility)
     */
    public function approve(Request $request)
    {
        $request->validate([
            'document_no' => 'required|string',
            'company' => 'required|string|in:TBI,CI',
            'category' => 'required|string',
            'remarks' => 'nullable|string|max:500',
        ]);

        return $this->bulkApprove(new Request([
            'document_numbers' => [$request->document_no],
            'company' => $request->company,
            'category' => $request->category,
            'remarks' => $request->remarks,
        ]));
    }

    /**
     * Single Reject (legacy - keep for backward compatibility)
     */
    public function reject(Request $request)
    {
        $request->validate([
            'document_no' => 'required|string',
            'company' => 'required|string|in:TBI,CI',
            'category' => 'required|string',
            'remarks' => 'nullable|string|max:500',
        ]);

        return $this->bulkReject(new Request([
            'document_numbers' => [$request->document_no],
            'company' => $request->company,
            'category' => $request->category,
            'remarks' => $request->remarks,
        ]));
    }
}