<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Services\DataSyncService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Laravel\Socialite\Facades\Socialite;

class AuthController extends Controller
{
    private $dataSyncService;

    public function __construct(DataSyncService $dataSyncService)
    {
        $this->dataSyncService = $dataSyncService;
    }

    /**
     * Show login page
     */
    public function showLogin()
    {
        return view('auth.login');
    }

    /**
     * Redirect to Microsoft OAuth
     */
    public function redirectToMicrosoft()
    {
        return Socialite::driver('microsoft')
            ->scopes(['openid', 'profile', 'email', 'User.Read'])
            ->redirect();
    }

    /**
     * Handle Microsoft OAuth Callback
     */
    public function handleMicrosoftCallback()
    {
        try {
            $microsoftUser = Socialite::driver('microsoft')->user();
            
            $email = $microsoftUser->getEmail();
            
            // Extract Approver ID from email
            $approverId = $this->extractApproverIdFromEmail($email);

            Log::info('Microsoft login successful', [
                'email' => $email,
                'approver_id' => $approverId
            ]);

            // Create or update user (email only, NO password)
            $user = User::updateOrCreate(
                ['email' => $email],
                [
                    'approver_id' => $approverId,
                    'name' => $microsoftUser->getName() ?? $approverId,
                    'role' => 'user',
                    'last_login' => now(),
                ]
            );

            // Login user
            Auth::login($user);

            // Redirect to data loading page
            return redirect()->route('data.load');

        } catch (\Exception $e) {
            Log::error('Microsoft login failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // JANGAN REDIRECT KE LOGIN - Biarkan user retry
            return redirect()->route('auth.microsoft');
        }
    }

    /**
     * Show data loading page
     */
    public function showDataLoad()
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        return view('auth.data-load');
    }

    /**
     * Load data via AJAX
     */
    public function loadData(Request $request)
    {
        try {
            $user = Auth::user();
            
            Log::info('Starting data load', [
                'user_id' => $user->id,
                'approver_id' => $user->approver_id
            ]);

            // Sync all data for this user
            $stats = $this->dataSyncService->syncUserData($user->approver_id);

            Log::info('Data load completed', $stats);

            return response()->json([
                'success' => true,
                'message' => 'Data loaded successfully',
                'stats' => $stats,
                'redirect' => route('dashboard')
            ]);

        } catch (\Exception $e) {
            Log::error('Data load failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Data load failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Admin login
     */
    public function adminLogin(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $user = User::where('email', $credentials['email'])
                    ->where('role', 'admin')
                    ->first();

        if ($user && Hash::check($credentials['password'], $user->password)) {
            Auth::login($user);
            
            Log::info('Admin login successful', ['email' => $user->email]);

            return redirect()->route('admin.dashboard');
        }

        return back()->withErrors([
            'email' => 'Invalid credentials or not an admin user.',
        ])->onlyInput('email');
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login');
    }

    /**
     * Extract Approver ID from email
     * Examples:
     * - surjantoro@citbi.onmicrosoft.com -> SURJANTORO
     * - michael.daniel@citbi.onmicrosoft.com -> MICHAEL DANIEL
     */
    private function extractApproverIdFromEmail(string $email): string
    {
        $localPart = explode('@', $email)[0];
        
        // Replace dots with spaces and convert to uppercase
        $approverId = str_replace('.', ' ', $localPart);
        $approverId = strtoupper($approverId);

        return $approverId;
    }
}