<?php

namespace App\Http\Controllers;

use App\Models\PoRekap;
use App\Models\PoDetail;
use App\Models\ApprovalHistory;
use App\Services\DataSyncService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class DashboardController extends Controller
{
    private $dataSyncService;

    public function __construct(DataSyncService $dataSyncService)
    {
        $this->dataSyncService = $dataSyncService;
    }

    /**
     * Show main dashboard (welcome page)
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get statistics untuk sidebar dan dashboard
        $stats = $this->getStats($user->approver_id);

        return view('dashboard.index', compact('stats'));
    }

    /**
     * Get statistics for user (EXCLUDE yang sudah di-approve/reject)
     */
    private function getStats($approverId)
    {
        // Get all approved/rejected document numbers
        $processedDocs = ApprovalHistory::where('approver_id', $approverId)
            ->pluck('document_no')
            ->toArray();

        return [
            'tbi_hirbnd' => PoRekap::where('company', 'TBI')
                ->where('category', 'HIRBND')
                ->where('approver_id', $approverId)
                ->whereNotIn('document_no', $processedDocs)
                ->count(),
                
            'tbi_ho' => PoRekap::where('company', 'TBI')
                ->where('category', 'HO')
                ->where('approver_id', $approverId)
                ->whereNotIn('document_no', $processedDocs)
                ->count(),
                
            'ci_ci' => PoRekap::where('company', 'CI')
                ->where('category', 'CI')
                ->where('approver_id', $approverId)
                ->whereNotIn('document_no', $processedDocs)
                ->count(),
                
            'ci_hirbc' => PoRekap::where('company', 'CI')
                ->where('category', 'HIRBC')
                ->where('approver_id', $approverId)
                ->whereNotIn('document_no', $processedDocs)
                ->count(),
                
            'ci_rbc' => PoRekap::where('company', 'CI')
                ->where('category', 'RBC')
                ->where('approver_id', $approverId)
                ->whereNotIn('document_no', $processedDocs)
                ->count(),
                
            'ci_ldr' => PoRekap::where('company', 'CI')
                ->where('category', 'LDR')
                ->where('approver_id', $approverId)
                ->whereNotIn('document_no', $processedDocs)
                ->count(),
        ];
    }

    /**
     * Show category data with PAGINATION (EXCLUDE yang sudah di-approve/reject)
     */
    public function showCategory(Request $request)
    {
        $user = Auth::user();
        $company = $request->route('company');
        $category = $request->route('category');
        
        // Get per page value from request (default 10)
        $perPage = $request->input('per_page', 10);
        
        // Validate per_page
        if (!in_array($perPage, [10, 20, 50, 'all'])) {
            $perPage = 10;
        }

        Log::info('Showing category', [
            'company' => $company,
            'category' => $category,
            'approver_id' => $user->approver_id,
            'per_page' => $perPage
        ]);

        // Get all approved/rejected document numbers for this user
        $processedDocs = ApprovalHistory::where('approver_id', $user->approver_id)
            ->pluck('document_no')
            ->toArray();

        // Build query
        $query = PoRekap::where('company', $company)
            ->where('category', $category)
            ->where('approver_id', $user->approver_id)
            ->whereNotIn('document_no', $processedDocs)
            ->orderBy('document_date', 'desc')
            ->orderBy('document_no', 'desc');

        // Apply pagination or get all
        if ($perPage === 'all') {
            $purchaseOrders = $query->get();
            // Create a fake paginator for consistency
            $purchaseOrders = new \Illuminate\Pagination\LengthAwarePaginator(
                $purchaseOrders,
                $purchaseOrders->count(),
                $purchaseOrders->count(),
                1,
                ['path' => $request->url(), 'query' => $request->query()]
            );
        } else {
            $purchaseOrders = $query->paginate($perPage)->appends($request->query());
        }

        // Get stats for sidebar
        $stats = $this->getStats($user->approver_id);

        return view('dashboard.category', compact('purchaseOrders', 'company', 'category', 'stats', 'perPage'));
    }

    /**
     * Show category with no API (CI Supplies)
     */
    public function showCategoryNoApi(Request $request)
    {
        $user = Auth::user();
        $company = $request->route('company');
        $category = $request->route('category');
        
        // Get stats for sidebar
        $stats = $this->getStats($user->approver_id);

        return view('dashboard.no-api', compact('company', 'category', 'stats'));
    }

    /**
 * Get PO Details (AJAX)
 */
public function getPoDetails(Request $request, string $company, string $category, string $documentNo)
{
    try {
        $user = Auth::user();

        // Check if already processed
        $alreadyProcessed = ApprovalHistory::where('document_no', $documentNo)
            ->where('approver_id', $user->approver_id)
            ->exists();

        if ($alreadyProcessed) {
            return response()->json([
                'success' => false,
                'message' => 'This PO has already been processed (approved/rejected)'
            ], 403);
        }

        // Verify ownership
        $rekap = PoRekap::where('company', $company)
            ->where('category', $category)
            ->where('document_no', $documentNo)
            ->where('approver_id', $user->approver_id)
            ->first();

        if (!$rekap) {
            return response()->json([
                'success' => false,
                'message' => 'Purchase Order not found or access denied'
            ], 404);
        }

        // Get details
        $details = PoDetail::where('company', $company)
            ->where('category', $category)
            ->where('document_no', $documentNo)
            ->get();

        return response()->json([
            'success' => true,
            'rekap' => $rekap,
            'details' => $details
        ]);

    } catch (\Exception $e) {
        \Log::error('Error loading PO details', [
            'company' => $company,
            'category' => $category,
            'document_no' => $documentNo,
            'error' => $e->getMessage(),
            'line' => $e->getLine(),
            'file' => $e->getFile()
        ]);

        return response()->json([
            'success' => false,
            'message' => 'Error loading details: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Show approval history
     */
    public function history()
    {
        $user = Auth::user();

        $history = ApprovalHistory::where('approver_id', $user->approver_id)
            ->orderBy('action_date', 'desc')
            ->paginate(50);

        // Get stats for sidebar
        $stats = $this->getStats($user->approver_id);

        return view('dashboard.history', compact('history', 'stats'));
    }

    /**
     * Manual sync (optional feature)
     */
    public function manualSync(Request $request)
    {
        try {
            $user = Auth::user();
            
            Log::info('Manual sync triggered', [
                'user_id' => $user->id,
                'approver_id' => $user->approver_id
            ]);

            $stats = $this->dataSyncService->syncUserData($user->approver_id);

            return response()->json([
                'success' => true,
                'message' => 'Data synchronized successfully',
                'stats' => $stats
            ]);

        } catch (\Exception $e) {
            Log::error('Manual sync failed', [
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Sync failed: ' . $e->getMessage()
            ], 500);
        }
    }
}