@extends('layouts.dashboard')

@section('title', $company . ' - ' . $category . ' - PO Approval System')
@section('page-title', $company . ' - ' . $category)

@section('content')
<style>
    /* ==================== STICKY HEADER ==================== */
    .category-header {
        background: white;
        border-radius: 10px;
        padding: 25px;
        margin-bottom: 25px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.08);
        display: flex;
        flex-direction: column;
        gap: 20px;
        position: sticky;
        top: 76px;
        z-index: 99;
    }

    .category-header-top {
        display: flex;
        align-items: center;
        justify-content: space-between;
    }

    .category-info h2 {
        font-size: 24px;
        color: var(--text-dark);
        margin-bottom: 5px;
    }

    .category-info p {
        color: var(--text-light);
        font-size: 14px;
    }

    .category-header-bottom {
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 20px;
        flex-wrap: wrap;
    }

    .left-controls {
        display: flex;
        align-items: center;
        gap: 15px;
        flex-wrap: wrap;
    }

    .page-selector {
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .page-selector label {
        font-size: 14px;
        color: var(--text-dark);
        font-weight: 600;
        white-space: nowrap;
    }

    .page-selector select {
        padding: 8px 12px;
        border: 1px solid var(--border-color);
        border-radius: 8px;
        font-size: 14px;
        background: white;
        cursor: pointer;
        min-width: 120px;
    }

    .action-buttons {
        display: flex;
        gap: 10px;
    }

    .search-box {
        position: relative;
        max-width: 350px;
        min-width: 250px;
        flex: 1;
    }

    .search-box input {
        width: 100%;
        padding: 10px 15px 10px 40px;
        border: 1px solid var(--border-color);
        border-radius: 8px;
        font-size: 14px;
    }

    .search-box svg {
        position: absolute;
        left: 12px;
        top: 50%;
        transform: translateY(-50%);
        color: var(--text-light);
    }

    /* ==================== PO CARDS ==================== */
    .po-cards-container {
        margin-bottom: 100px;
    }

    .po-card {
        background: white;
        border-radius: 12px;
        margin-bottom: 15px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        overflow: hidden;
        transition: all 0.3s ease;
    }

    .po-card:hover {
        box-shadow: 0 4px 16px rgba(0,0,0,0.12);
    }

    .po-card-header {
        padding: 20px;
        display: flex;
        align-items: center;
        gap: 15px;
        cursor: pointer;
        background: white;
        border-bottom: 1px solid transparent;
        transition: all 0.3s ease;
    }

    .po-card-header:hover {
        background: #f8f9fa;
    }

    .po-card.expanded .po-card-header {
        border-bottom-color: var(--border-color);
        background: #f8f9fa;
    }

    .po-checkbox {
        width: 20px;
        height: 20px;
        cursor: pointer;
        margin: 0;
    }

    .expand-icon-wrapper {
        width: 32px;
        height: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: #e3f2fd;
        border-radius: 6px;
        transition: all 0.3s ease;
    }

    .po-card.expanded .expand-icon-wrapper {
        background: #2196F3;
    }

    .expand-icon-wrapper svg {
        transition: transform 0.3s ease;
        fill: #2196F3;
    }

    .po-card.expanded .expand-icon-wrapper svg {
        transform: rotate(180deg);
        fill: white;
    }

    .po-card-info {
        flex: 1;
        display: grid;
        grid-template-columns: auto 1fr 150px 180px 180px 120px 100px;
        gap: 20px;
        align-items: center;
    }

    .po-info-item {
        min-width: 0;
    }

    .po-info-label {
        font-size: 11px;
        color: var(--text-light);
        font-weight: 600;
        text-transform: uppercase;
        margin-bottom: 4px;
    }

    .po-info-value {
        font-size: 14px;
        color: var(--text-dark);
        font-weight: 600;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
    }

    .po-info-value.document-no {
        color: #2196F3;
        font-size: 16px;
    }

    .po-info-value.amount {
        color: #2c7a3f;
        font-weight: 700;
    }

    .status-badge {
        display: inline-block;
        padding: 4px 12px;
        border-radius: 12px;
        font-size: 12px;
        font-weight: 600;
    }

    .status-badge.open {
        background: #fff3cd;
        color: #856404;
    }

    .status-badge.released {
        background: #d4edda;
        color: #155724;
    }

    /* ==================== CARD BODY ==================== */
    .po-card-body {
        max-height: 0;
        overflow: hidden;
        transition: max-height 0.3s ease;
    }

    .po-card.expanded .po-card-body {
        max-height: 2000px;
    }

    .po-card-body-content {
        padding: 20px;
        background: #f8f9fa;
    }

    /* ==================== ITEMS TABLE ==================== */
    .items-section {
        background: white;
        padding: 20px;
        border-radius: 8px;
    }

    .items-section-header {
        display: flex;
        align-items: center;
        gap: 10px;
        margin-bottom: 15px;
        padding-bottom: 10px;
        border-bottom: 2px solid var(--border-color);
    }

    .items-section-header svg {
        fill: #2196F3;
    }

    .items-section-header h4 {
        font-size: 14px;
        font-weight: 700;
        color: var(--text-dark);
        margin: 0;
    }

    .items-count {
        background: #2196F3;
        color: white;
        padding: 2px 10px;
        border-radius: 12px;
        font-size: 12px;
        font-weight: 600;
    }

    .items-table {
        width: 100%;
        border-collapse: collapse;
    }

    .items-table thead {
        background: #f8f9fa;
    }

    .items-table th {
        padding: 12px;
        text-align: left;
        font-size: 11px;
        font-weight: 700;
        color: var(--text-dark);
        text-transform: uppercase;
        border-bottom: 2px solid var(--border-color);
    }

    .items-table td {
        padding: 12px;
        font-size: 13px;
        border-bottom: 1px solid var(--border-color);
    }

    .items-table tbody tr:last-child td {
        border-bottom: none;
    }

    /* ==================== PAGINATION ==================== */
    .pagination-wrapper {
        background: white;
        padding: 20px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.08);
        margin-bottom: 20px;
        display: flex;
        align-items: center;
        justify-content: space-between;
        flex-wrap: wrap;
        gap: 15px;
    }

    .pagination-info {
        font-size: 14px;
        color: var(--text-dark);
    }

    .pagination-links {
        display: flex;
        gap: 5px;
        flex-wrap: wrap;
    }

    .pagination-links a,
    .pagination-links span {
        padding: 8px 12px;
        border: 1px solid var(--border-color);
        border-radius: 6px;
        font-size: 14px;
        text-decoration: none;
        color: var(--text-dark);
        transition: all 0.3s ease;
        min-width: 40px;
        text-align: center;
    }

    .pagination-links a:hover {
        background: #2196F3;
        color: white;
        border-color: #2196F3;
    }

    .pagination-links .active {
        background: #2196F3;
        color: white;
        border-color: #2196F3;
        font-weight: 600;
    }

    .pagination-links .disabled {
        opacity: 0.5;
        cursor: not-allowed;
        pointer-events: none;
    }

    /* ==================== STICKY ACTION BAR ==================== */
    .sticky-action-bar {
        position: fixed;
        bottom: 0;
        left: 280px;
        right: 0;
        background: white;
        padding: 15px 30px;
        box-shadow: 0 -2px 10px rgba(0,0,0,0.1);
        display: flex;
        align-items: center;
        justify-content: space-between;
        z-index: 999;
        transition: left 0.3s ease;
    }

    .sidebar.collapsed + .main-content .sticky-action-bar {
        left: 80px;
    }

    .selected-info {
        font-size: 14px;
        color: var(--text-dark);
        font-weight: 600;
    }

    .selected-count {
        color: var(--primary-color);
        font-size: 18px;
    }

    .action-bar-buttons {
        display: flex;
        gap: 10px;
        align-items: center;
    }

    .remarks-section {
        flex: 1;
        max-width: 400px;
        margin-right: 15px;
    }

    .remarks-section textarea {
        width: 100%;
        padding: 8px 10px;
        border: 1px solid var(--border-color);
        border-radius: 5px;
        resize: none;
        font-size: 13px;
        height: 45px;
    }

    .loading-overlay {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(255,255,255,0.9);
        z-index: 10000;
        align-items: center;
        justify-content: center;
    }

    .loading-overlay.active {
        display: flex;
    }

    .no-data {
        text-align: center;
        padding: 60px 20px;
        color: var(--text-light);
    }

    .no-data svg {
        width: 80px;
        height: 80px;
        margin-bottom: 20px;
        opacity: 0.3;
    }

    /* ==================== RESPONSIVE ==================== */
    @media (max-width: 1400px) {
        .po-card-info {
            grid-template-columns: auto 1fr;
            gap: 15px;
        }

        .po-info-item:not(:first-child) {
            grid-column: span 2;
        }
    }

    @media (max-width: 1024px) {
        .category-header {
            top: 60px;
        }

        .sticky-action-bar {
            left: 0;
        }
    }

    @media (max-width: 768px) {
        .category-header-bottom {
            flex-direction: column;
            align-items: stretch;
        }

        .left-controls {
            width: 100%;
            flex-direction: column;
            align-items: stretch;
        }

        .page-selector {
            width: 100%;
        }

        .page-selector select {
            width: 100%;
        }

        .action-buttons {
            width: 100%;
        }

        .action-buttons .btn {
            flex: 1;
        }

        .search-box {
            width: 100%;
            max-width: none;
        }

        .po-card-info {
            grid-template-columns: 1fr;
        }

        .sticky-action-bar {
            flex-direction: column;
            gap: 10px;
            padding: 10px 15px;
        }

        .remarks-section {
            max-width: 100%;
            width: 100%;
            margin-right: 0;
        }

        .action-bar-buttons {
            width: 100%;
        }

        .action-bar-buttons .btn {
            flex: 1;
        }

        .pagination-wrapper {
            flex-direction: column;
            align-items: flex-start;
        }

        .pagination-links {
            width: 100%;
            justify-content: center;
        }
    }
</style>

<div class="category-header">
    <div class="category-header-top">
        <div class="category-info">
            <h2>{{ $company }} - {{ $category }}</h2>
            <p>Showing {{ $purchaseOrders->total() }} purchase orders awaiting approval</p>
        </div>
    </div>
    <div class="category-header-bottom">
        <!-- LEFT SIDE: Page Selector + Action Buttons -->
        <div class="left-controls">
            <div class="page-selector">
                <label>Show:</label>
                <select id="perPageSelect" onchange="changePerPage(this.value)">
                    <option value="10" {{ $perPage == 10 ? 'selected' : '' }}>10 per page</option>
                    <option value="20" {{ $perPage == 20 ? 'selected' : '' }}>20 per page</option>
                    <option value="50" {{ $perPage == 50 ? 'selected' : '' }}>50 per page</option>
                    <option value="all" {{ $perPage == 'all' ? 'selected' : '' }}>All</option>
                </select>
            </div>
            
            <div class="action-buttons">
                <button class="btn btn-secondary" onclick="selectAll()">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M19 3H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm0 16H5V5h14v14zM17.99 9l-1.41-1.42-6.59 6.59-2.58-2.57-1.42 1.41 4 3.99z" fill="currentColor"/>
                    </svg>
                    Select All
                </button>
                <button class="btn btn-secondary" id="toggleExpandBtn" onclick="toggleExpandCollapse()">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" id="expandIcon">
                        <path d="M12 5.83L15.17 9l1.41-1.41L12 3 7.41 7.59 8.83 9 12 5.83zm0 12.34L8.83 15l-1.41 1.41L12 21l4.59-4.59L15.17 15 12 18.17z" fill="currentColor"/>
                    </svg>
                    <span id="expandText">Expand All</span>
                </button>
            </div>
        </div>

        <!-- RIGHT SIDE: Search Box -->
        <div class="search-box">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z" fill="currentColor"/>
            </svg>
            <input type="text" id="searchInput" placeholder="Search by Document No, Vendor...">
        </div>
    </div>
</div>

<div class="po-cards-container">
    @if($purchaseOrders->count() > 0)
        @foreach($purchaseOrders as $po)
        <div class="po-card" data-doc-no="{{ $po->document_no }}" data-company="{{ $po->company }}" data-category="{{ $po->category }}">
            <div class="po-card-header">
                <input type="checkbox" class="po-checkbox" value="{{ $po->document_no }}" onchange="updateSelectedCount()" onclick="event.stopPropagation()">
                
                <div class="expand-icon-wrapper" onclick="toggleCard('{{ $po->document_no }}', '{{ $po->company }}', '{{ $po->category }}')">
                    <svg width="16" height="16" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                        <path d="M7.41 8.59L12 13.17l4.59-4.58L18 10l-6 6-6-6 1.41-1.41z"/>
                    </svg>
                </div>

                <div class="po-card-info" onclick="toggleCard('{{ $po->document_no }}', '{{ $po->company }}', '{{ $po->category }}')">
                    <div class="po-info-item">
                        <div class="po-info-label">Document No</div>
                        <div class="po-info-value document-no">{{ $po->document_no }}</div>
                    </div>
                    <div class="po-info-item">
                        <div class="po-info-label">Vendor</div>
                        <div class="po-info-value">{{ $po->buy_from_vendor_name }}</div>
                    </div>
                    <div class="po-info-item">
                        <div class="po-info-label">Document Date</div>
                        <div class="po-info-value">{{ $po->document_date ? $po->document_date->format('d M Y') : '-' }}</div>
                    </div>
                    <div class="po-info-item">
                        <div class="po-info-label">Amount</div>
                        <div class="po-info-value amount">IDR {{ number_format($po->amount, 0, ',', '.') }}</div>
                    </div>
                    <div class="po-info-item">
                        <div class="po-info-label">Amount (Inc VAT)</div>
                        <div class="po-info-value amount">IDR {{ number_format($po->amount_including_vat, 0, ',', '.') }}</div>
                    </div>
                    <div class="po-info-item">
                        <div class="po-info-label">Status</div>
                        <div class="po-info-value">
                            <span class="status-badge {{ strtolower($po->status) }}">{{ $po->status }}</span>
                        </div>
                    </div>
                    <div class="po-info-item">
                        <div class="po-info-label">Type</div>
                        <div class="po-info-value">{{ $po->document_type }}</div>
                    </div>
                </div>
            </div>

            <div class="po-card-body" id="body-{{ $po->document_no }}">
                <div class="po-card-body-content" id="content-{{ $po->document_no }}">
                    <div style="text-align: center; padding: 20px;">
                        <div class="spinner"></div>
                        <p style="margin-top: 10px;">Loading details...</p>
                    </div>
                </div>
            </div>
        </div>
        @endforeach

        <!-- PAGINATION -->
        @if($perPage !== 'all' && $purchaseOrders->hasPages())
        <div class="pagination-wrapper">
            <div class="pagination-info">
                Showing {{ $purchaseOrders->firstItem() }} to {{ $purchaseOrders->lastItem() }} of {{ $purchaseOrders->total() }} results
            </div>
            <div class="pagination-links">
                {{-- Previous Page Link --}}
                @if ($purchaseOrders->onFirstPage())
                    <span class="disabled">« Previous</span>
                @else
                    <a href="{{ $purchaseOrders->previousPageUrl() }}">« Previous</a>
                @endif

                {{-- Pagination Elements --}}
                @foreach ($purchaseOrders->links()->elements[0] as $page => $url)
                    @if ($page == $purchaseOrders->currentPage())
                        <span class="active">{{ $page }}</span>
                    @else
                        <a href="{{ $url }}">{{ $page }}</a>
                    @endif
                @endforeach

                {{-- Next Page Link --}}
                @if ($purchaseOrders->hasMorePages())
                    <a href="{{ $purchaseOrders->nextPageUrl() }}">Next »</a>
                @else
                    <span class="disabled">Next »</span>
                @endif
            </div>
        </div>
        @endif
    @else
        <div class="no-data">
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                <path d="M19 3h-4.18C14.4 1.84 13.3 1 12 1c-1.3 0-2.4.84-2.82 2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm-7 0c.55 0 1 .45 1 1s-.45 1-1 1-1-.45-1-1 .45-1 1-1zm2 14H7v-2h7v2zm3-4H7v-2h10v2zm0-4H7V7h10v2z"/>
            </svg>
            <h3>No Purchase Orders Found</h3>
            <p>There are no purchase orders awaiting your approval in this category.</p>
        </div>
    @endif
</div>

<!-- Sticky Action Bar -->
<div class="sticky-action-bar">
    <div class="selected-info">
        Selected: <span class="selected-count" id="selectedCount">0</span> PO(s)
    </div>
    <div class="action-bar-buttons">
        <div class="remarks-section">
            <textarea id="bulkRemarks" placeholder="Remarks (optional)..."></textarea>
        </div>
        <button class="btn btn-danger" onclick="bulkAction('reject')" id="btnReject">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z" fill="currentColor"/>
            </svg>
            Reject Selected
        </button>
        <button class="btn btn-success" onclick="bulkAction('approve')" id="btnApprove">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z" fill="currentColor"/>
            </svg>
            Approve Selected
        </button>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay">
    <div class="loading-content">
        <div class="spinner"></div>
        <p>Processing request...</p>
    </div>
</div>

@push('scripts')
<script>
    let loadedDetails = {};
    let isAllExpanded = false;
    const company = '{{ $company }}';
    const category = '{{ $category }}';
    const currentPerPage = '{{ $perPage }}';

    // Change Per Page
    function changePerPage(perPage) {
        const url = new URL(window.location.href);
        url.searchParams.set('per_page', perPage);
        url.searchParams.delete('page');
        window.location.href = url.toString();
    }

    // Search functionality
    document.getElementById('searchInput').addEventListener('keyup', function(e) {
        const searchTerm = e.target.value.toLowerCase();
        const cards = document.querySelectorAll('.po-card');
        
        cards.forEach(card => {
            const text = card.textContent.toLowerCase();
            card.style.display = text.includes(searchTerm) ? '' : 'none';
        });
    });

    // Select All
    function selectAll() {
        const checkboxes = document.querySelectorAll('.po-checkbox');
        const allChecked = Array.from(checkboxes).every(cb => cb.checked);
        
        checkboxes.forEach(cb => {
            cb.checked = !allChecked;
        });
        
        updateSelectedCount();
    }

    // Update Selected Count
    function updateSelectedCount() {
        const checkboxes = document.querySelectorAll('.po-checkbox:checked');
        document.getElementById('selectedCount').textContent = checkboxes.length;
    }

    // Toggle Expand/Collapse (SINGLE BUTTON)
    function toggleExpandCollapse() {
        const cards = document.querySelectorAll('.po-card');
        const expandText = document.getElementById('expandText');
        const expandIcon = document.getElementById('expandIcon');
        
        if (isAllExpanded) {
            // Collapse All
            cards.forEach(card => {
                card.classList.remove('expanded');
            });
            expandText.textContent = 'Expand All';
            expandIcon.innerHTML = '<path d="M12 5.83L15.17 9l1.41-1.41L12 3 7.41 7.59 8.83 9 12 5.83zm0 12.34L8.83 15l-1.41 1.41L12 21l4.59-4.59L15.17 15 12 18.17z" fill="currentColor"/>';
            isAllExpanded = false;
        } else {
            // Expand All
            cards.forEach(card => {
                if (!card.classList.contains('expanded')) {
                    const docNo = card.dataset.docNo;
                    const company = card.dataset.company;
                    const category = card.dataset.category;
                    
                    card.classList.add('expanded');
                    
                    if (!loadedDetails[docNo]) {
                        loadPoDetails(docNo, company, category);
                    }
                }
            });
            expandText.textContent = 'Collapse All';
            expandIcon.innerHTML = '<path d="M7.41 18.59L8.83 20 12 16.83 15.17 20l1.41-1.41L12 14l-4.59 4.59zm9.18-13.18L15.17 4 12 7.17 8.83 4 7.41 5.41 12 10l4.59-4.59z" fill="currentColor"/>';
            isAllExpanded = true;
        }
    }

    // Toggle Card
    function toggleCard(docNo, company, category) {
        const card = document.querySelector(`[data-doc-no="${docNo}"]`);
        card.classList.toggle('expanded');
        
        if (!loadedDetails[docNo] && card.classList.contains('expanded')) {
            loadPoDetails(docNo, company, category);
        }
    }

    // Load PO Details - FIX FOR HOSTING
    function loadPoDetails(docNo, company, category) {
        const contentDiv = document.getElementById(`content-${docNo}`);
        
        // Generate URL using Laravel route helper
        const detailUrl = "{{ route('po.details', ['company' => '__COMPANY__', 'category' => '__CATEGORY__', 'documentNo' => '__DOCNO__']) }}"
            .replace('__COMPANY__', encodeURIComponent(company))
            .replace('__CATEGORY__', encodeURIComponent(category))
            .replace('__DOCNO__', encodeURIComponent(docNo));
        
        console.log('Loading PO details from:', detailUrl); // Debug log
        
        fetch(detailUrl, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': window.csrfToken || ''
            },
            credentials: 'same-origin'
        })
        .then(response => {
            console.log('Response status:', response.status); // Debug log
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const contentType = response.headers.get('content-type');
            if (!contentType || !contentType.includes('application/json')) {
                return response.text().then(text => {
                    console.error('Non-JSON response:', text.substring(0, 200));
                    throw new Error('Server returned HTML instead of JSON. Please check server logs.');
                });
            }
            
            return response.json();
        })
        .then(data => {
            console.log('PO details loaded:', data); // Debug log
            
            if (data.success) {
                displayPoDetails(docNo, data.rekap, data.details);
                loadedDetails[docNo] = true;
            } else {
                contentDiv.innerHTML = `
                    <div class="alert alert-danger">
                        ${data.message || 'Failed to load details'}
                    </div>
                `;
            }
        })
        .catch(error => {
            console.error('Error loading PO details:', error); // Debug log
            
            contentDiv.innerHTML = `
                <div class="alert alert-danger">
                    Error loading details: ${error.message}
                </div>
            `;
        });
    }

    // Display PO Details
    function displayPoDetails(docNo, rekap, details) {
        const contentDiv = document.getElementById(`content-${docNo}`);
        
        let itemsHtml = '';
        if (details && details.length > 0) {
            itemsHtml = `
                <div class="items-section">
                    <div class="items-section-header">
                        <svg width="20" height="20" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                            <path d="M19 3H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm0 16H5V5h14v14z" fill="currentColor"/>
                        </svg>
                        <h4>ITEMS</h4>
                        <span class="items-count">${details.length}</span>
                    </div>
                    <table class="items-table">
                        <thead>
                            <tr>
                                <th>ITEM</th>
                                <th>DESCRIPTION</th>
                                <th>QTY</th>
                                <th>UNIT</th>
                                <th>PRICE</th>
                                <th>AMOUNT</th>
                                <th>DIMENSION</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${details.map((item, index) => `
                                <tr>
                                    <td><strong>${item.item_no || (index + 1)}</strong></td>
                                    <td>${item.description || '-'}</td>
                                    <td>${parseFloat(item.quantity_base).toLocaleString('id-ID')}</td>
                                    <td><strong>${item.unit_of_measure_code || 'PCS'}</strong></td>
                                    <td>IDR ${(item.quantity_base > 0 ? parseFloat(item.amount / item.quantity_base) : 0).toLocaleString('id-ID', {minimumFractionDigits: 0})}</td>
                                    <td><strong>IDR ${parseFloat(item.amount).toLocaleString('id-ID', {minimumFractionDigits: 0})}</strong></td>
                                    <td>${item.shortcut_dimension_1_code || '-'}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
            `;
        } else {
            itemsHtml = '<div class="alert alert-info">No items found for this PO.</div>';
        }

        contentDiv.innerHTML = itemsHtml;
    }

    // Bulk Action - FIX FOR HOSTING
    function bulkAction(action) {
        const checkboxes = document.querySelectorAll('.po-checkbox:checked');
        
        if (checkboxes.length === 0) {
            showToast('Please select at least one PO', 'error');
            return;
        }

        const documentNumbers = Array.from(checkboxes).map(cb => cb.value);
        const remarks = document.getElementById('bulkRemarks').value;
        const actionText = action === 'approve' ? 'APPROVE' : 'REJECT';
        
        if (!confirm(`Are you sure you want to ${actionText} ${documentNumbers.length} PO(s)?`)) {
            return;
        }

        const loadingOverlay = document.getElementById('loadingOverlay');
        loadingOverlay.classList.add('active');

        // Generate URL using Laravel route helper
        const url = action === 'approve' 
            ? "{{ route('po.bulk.approve') }}" 
            : "{{ route('po.bulk.reject') }}";
        
        console.log('Bulk action URL:', url); // Debug log
        console.log('Payload:', {
            document_numbers: documentNumbers,
            company: company,
            category: category,
            remarks: remarks
        }); // Debug log

        ajaxRequest(url, 'POST', {
            document_numbers: documentNumbers,
            company: company,
            category: category,
            remarks: remarks
        })
        .then(data => {
            console.log('Bulk action response:', data); // Debug log
            
            loadingOverlay.classList.remove('active');
            if (data.success) {
                showToast(data.message, 'success');
                setTimeout(() => location.reload(), 2000);
            } else {
                showToast(data.message || 'Action failed', 'error');
            }
        })
        .catch(error => {
            console.error('Bulk action error:', error); // Debug log
            
            loadingOverlay.classList.remove('active');
            showToast('Error: ' + error.message, 'error');
        });
    }
</script>
@endpush
@endsection