document.addEventListener('DOMContentLoaded', function() {
    const sidebar = document.getElementById('sidebar');
    const hamburger = document.getElementById('hamburger');
    const mainContent = document.getElementById('mainContent');

    // Hamburger Toggle (DESKTOP + MOBILE) - NO X ANIMATION
    if (hamburger) {
        hamburger.addEventListener('click', function(e) {
            e.stopPropagation();
            
            // Check screen size
            if (window.innerWidth > 1024) {
                // DESKTOP: Toggle collapsed class (hide completely)
                sidebar.classList.toggle('collapsed');
                
                // Save state to localStorage
                const isCollapsed = sidebar.classList.contains('collapsed');
                localStorage.setItem('sidebarCollapsed', isCollapsed);
            } else {
                // MOBILE: Toggle mobile-open class
                sidebar.classList.toggle('mobile-open');
                hamburger.classList.toggle('active');
            }
        });
    }

    // Restore sidebar state from localStorage (DESKTOP ONLY)
    if (window.innerWidth > 1024) {
        if (localStorage.getItem('sidebarCollapsed') === 'true') {
            sidebar.classList.add('collapsed');
        }
    }

    // User Dropdown Toggle
    const userDropdownToggle = document.getElementById('userDropdownToggle');
    const userDropdownWrapper = document.querySelector('.user-dropdown-wrapper');
    const userDropdownMenu = document.getElementById('userDropdownMenu');

    if (userDropdownToggle) {
        userDropdownToggle.addEventListener('click', function(e) {
            e.stopPropagation();
            userDropdownWrapper.classList.toggle('active');
        });
    }

    // Close user dropdown when clicking outside
    document.addEventListener('click', function(e) {
        if (userDropdownWrapper && !userDropdownWrapper.contains(e.target)) {
            userDropdownWrapper.classList.remove('active');
        }
    });

    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', function(e) {
        if (window.innerWidth <= 1024) {
            if (!sidebar.contains(e.target) && !hamburger.contains(e.target)) {
                sidebar.classList.remove('mobile-open');
                hamburger.classList.remove('active');
            }
        }
    });

    // Prevent sidebar close when clicking inside
    if (sidebar) {
        sidebar.addEventListener('click', function(e) {
            e.stopPropagation();
        });
    }

    // Prevent dropdown close when clicking inside
    if (userDropdownMenu) {
        userDropdownMenu.addEventListener('click', function(e) {
            e.stopPropagation();
        });
    }

    // Nav Group Toggle
    const navGroups = document.querySelectorAll('.nav-group-header');
    navGroups.forEach(header => {
        header.addEventListener('click', function(e) {
            e.preventDefault();
            const group = this.parentElement;
            group.classList.toggle('open');
            
            // Save state to localStorage
            const groupName = this.querySelector('span').textContent.trim();
            const isOpen = group.classList.contains('open');
            localStorage.setItem(`navGroup_${groupName}`, isOpen);
        });
    });

    // Restore nav group states from localStorage
    navGroups.forEach(header => {
        const groupName = header.querySelector('span').textContent.trim();
        const savedState = localStorage.getItem(`navGroup_${groupName}`);
        
        if (savedState !== null) {
            const group = header.parentElement;
            if (savedState === 'true') {
                group.classList.add('open');
            } else {
                group.classList.remove('open');
            }
        }
    });

    // Open active nav group by default
    const activeNavItem = document.querySelector('.nav-group-items .nav-item.active');
    if (activeNavItem) {
        const activeGroup = activeNavItem.closest('.nav-group');
        if (activeGroup) {
            activeGroup.classList.add('open');
        }
    }

    // Close mobile sidebar when clicking nav links
    const navLinks = document.querySelectorAll('.nav-item');
    navLinks.forEach(link => {
        link.addEventListener('click', function() {
            if (window.innerWidth <= 1024) {
                sidebar.classList.remove('mobile-open');
                hamburger.classList.remove('active');
            }
        });
    });
});

/**
 * Window resize handler
 */
let resizeTimer;
window.addEventListener('resize', function() {
    clearTimeout(resizeTimer);
    resizeTimer = setTimeout(function() {
        const sidebar = document.getElementById('sidebar');
        const hamburger = document.getElementById('hamburger');
        const userDropdownWrapper = document.querySelector('.user-dropdown-wrapper');
        
        // Reset mobile menu state when resizing to desktop
        if (window.innerWidth > 1024) {
            sidebar.classList.remove('mobile-open');
            
            // Restore collapsed state from localStorage for desktop
            const isCollapsed = localStorage.getItem('sidebarCollapsed') === 'true';
            if (isCollapsed) {
                sidebar.classList.add('collapsed');
            } else {
                sidebar.classList.remove('collapsed');
            }
        } else {
            // Mobile: Remove collapsed class
            sidebar.classList.remove('collapsed');
        }
        
        // Close user dropdown on resize
        if (userDropdownWrapper) {
            userDropdownWrapper.classList.remove('active');
        }
    }, 250);
});

/**
 * Close dropdowns on ESC key
 */
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        const userDropdownWrapper = document.querySelector('.user-dropdown-wrapper');
        if (userDropdownWrapper) {
            userDropdownWrapper.classList.remove('active');
        }
    }
});