<?php

namespace App\Services\BusinessCentral;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class HirbcApiService extends BaseApiService
{
    /**
     * Fetch room data for specific month
     */
    public function fetchRoomData($month, $year)
    {
        try {
            Log::info("HIRBC: Starting fetch for {$month} {$year}");
            
            // Calculate date range
            $monthNumber = date('n', strtotime($month));
            $startDate = date('Y-m-d', strtotime("{$year}-{$monthNumber}-01"));
            
            // Calculate end date (first day of next month)
            if ($monthNumber == 12) {
                $endDate = date('Y-m-d', strtotime(($year + 1) . "-01-01"));
            } else {
                $endDate = date('Y-m-d', strtotime("{$year}-" . ($monthNumber + 1) . "-01"));
            }
            
            Log::info("HIRBC: Fetching with filter", [
                'month' => $month,
                'year' => $year,
                'date_range' => "{$startDate} to {$endDate}"
            ]);
            
            // Get endpoint from config
            $baseEndpoint = config('businesscentral.hirbc.endpoints.room');
            
            // Append date filter
            $fullEndpoint = $baseEndpoint . " and Posting_Date ge {$startDate} and Posting_Date lt {$endDate}";
            
            // Call API
            $response = $this->get($fullEndpoint);
            
            if (isset($response['value']) && is_array($response['value'])) {
                Log::info("HIRBC: ✅ Fetched {$month} {$year}", [
                    'records' => count($response['value'])
                ]);
                
                return [
                    'success' => true,
                    'data' => $response['value'],
                    'records' => count($response['value'])
                ];
            }
            
            Log::warning("HIRBC: No data returned for {$month} {$year}");
            
            return [
                'success' => true,
                'data' => [],
                'records' => 0
            ];
            
        } catch (\Exception $e) {
            Log::error("HIRBC: ❌ Fetch failed for {$month} {$year}", [
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'data' => [],
                'records' => 0,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Fetch all months data
     */
    public function fetchAllMonthsData($year)
    {
        $results = [];
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        foreach ($months as $month) {
            $results[$month] = $this->fetchRoomData($month, $year);
        }
        
        return $results;
    }

    /**
     * Fetch OOO data
     */
    public function fetchOOOData()
    {
        try {
            Log::info("HIRBC: Fetching OOO data", [
                'filter' => "Completed_On eq 0001-01-01"
            ]);
            
            // Get endpoint from config
            $endpoint = config('businesscentral.hirbc.endpoints.ooo');
            
            // Append completed filter
            $fullEndpoint = $endpoint . " and Completed_On eq 0001-01-01";
            
            // Call API
            $response = $this->get($fullEndpoint);
            
            if (isset($response['value']) && is_array($response['value'])) {
                Log::info("HIRBC: ✅ Fetched OOO data", [
                    'records' => count($response['value'])
                ]);
                
                return [
                    'success' => true,
                    'data' => $response['value'],
                    'records' => count($response['value'])
                ];
            }
            
            Log::warning("HIRBC: No OOO data returned");
            
            return [
                'success' => true,
                'data' => [],
                'records' => 0
            ];
            
        } catch (\Exception $e) {
            Log::error("HIRBC: ❌ OOO fetch failed", [
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'data' => [],
                'records' => 0,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Save room data to database
     */
    public function saveRoomData($month, $data)
    {
        try {
            if (empty($data)) {
                Log::info("HIRBC: No data to save for {$month}");
                return [
                    'success' => true,
                    'records' => 0
                ];
            }

            $tableName = "{$month}_hirbc";
            
            Log::info("HIRBC: Saving {$month} data", [
                'table' => $tableName,
                'records' => count($data)
            ]);

            DB::beginTransaction();

            // ✅ FIX: Gunakan delete() bukan truncate()
            DB::table($tableName)->delete();
            
            // Batch insert
            $batchSize = config('businesscentral.batch_size.room_data', 500);
            $chunks = array_chunk($data, $batchSize);
            
            foreach ($chunks as $index => $chunk) {
                $insertData = [];
                
                foreach ($chunk as $record) {
                    $insertData[] = [
                        'Business_Unit' => $record['Business_Unit'] ?? null,
                        'Room' => $record['Room'] ?? null,
                        'Posting_Date' => $record['Posting_Date'] ?? null,
                        'Market_Code' => $record['Market_Code'] ?? null,
                        'TRX_Code' => $record['TRX_Code'] ?? null,
                        'Amount_1' => $record['Amount_1'] ?? 0,
                        'Description' => $record['Description'] ?? null,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }
                
                DB::table($tableName)->insert($insertData);
                
                Log::info("HIRBC: Saved chunk " . ($index + 1) . "/" . count($chunks), [
                    'records' => count($insertData)
                ]);
            }

            DB::commit();

            Log::info("HIRBC: ✅ Successfully saved {$month} data", [
                'records' => count($data)
            ]);

            return [
                'success' => true,
                'records' => count($data)
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error("HIRBC: ❌ Save failed for {$month}", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'records' => 0,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
 * Save OOO data to database
 */
public function saveOOOData($data)
{
    try {
        if (empty($data)) {
            Log::info("HIRBC: No OOO data to save");
            return [
                'success' => true,
                'records' => 0
            ];
        }

        Log::info("HIRBC: Saving OOO data", [
            'records' => count($data)
        ]);

        DB::beginTransaction();

        // ✅ FIX: Gunakan delete() bukan truncate()
        DB::table('ooo_hirbc')->delete();
        
        // Batch insert
        $batchSize = config('businesscentral.batch_size.ooo_data', 500);
        $chunks = array_chunk($data, $batchSize);
        
        foreach ($chunks as $index => $chunk) {
            $insertData = [];
            
            foreach ($chunk as $record) {
                $insertData[] = [
                    'odata_etag' => $record['@odata.etag'] ?? null,
                    'Business_Unit' => $record['Business_Unit'] ?? null,
                    'AuxiliaryIndex1' => $record['AuxiliaryIndex1'] ?? null,
                    'Begin_Date' => $record['Begin_Date'] ?? null,
                    'Completed_On' => $record['Completed_On'] ?? null,
                    'End_Date' => $record['End_Date'] ?? null, // ✅ ADDED!
                    'Label' => $record['Label'] ?? null, // ✅ ADDED!
                    'Reason_Code' => $record['Reason_Code'] ?? null, // ✅ ADDED!
                    'Reason_Desc' => $record['Reason_Desc'] ?? null, // ✅ ADDED!
                    'Room_No' => $record['Room_No'] ?? null,
                    'Room_Class' => $record['Room_Class'] ?? null, // ✅ ADDED!
                    'Room_Status' => $record['Room_Status'] ?? null, // ✅ ADDED!
                    'Repair_Remark' => $record['Repair_Remark'] ?? null,
                    'Return_Status' => $record['Return_Status'] ?? null, // ✅ ADDED!
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
            
            DB::table('ooo_hirbc')->insert($insertData);
            
            Log::info("HIRBC: Saved OOO chunk " . ($index + 1) . "/" . count($chunks), [
                'records' => count($insertData)
            ]);
        }

        DB::commit();

        Log::info("HIRBC: ✅ Successfully saved OOO data", [
            'records' => count($data)
        ]);

        return [
            'success' => true,
            'records' => count($data)
        ];

    } catch (\Exception $e) {
        DB::rollBack();
        
        Log::error("HIRBC: ❌ OOO save failed", [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);

        return [
            'success' => false,
            'records' => 0,
            'error' => $e->getMessage()
        ];
    }
}

    /**
     * Refresh all data (12 months + OOO)
     */
    public function refreshAllData($year)
    {
        $startTime = microtime(true);
        
        Log::info("HIRBC: Starting data refresh for year {$year}");
        
        $results = [
            'success' => true,
            'months' => [],
            'ooo' => [],
            'total_records' => 0,
            'errors' => []
        ];

        // Fetch and save monthly data
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];

        foreach ($months as $month) {
            // Fetch data
            $fetchResult = $this->fetchRoomData($month, $year);
            
            if ($fetchResult['success'] && !empty($fetchResult['data'])) {
                // Save data
                $saveResult = $this->saveRoomData($month, $fetchResult['data']);
                
                $results['months'][$month] = [
                    'success' => $saveResult['success'],
                    'records' => $saveResult['records']
                ];
                
                if (!$saveResult['success']) {
                    $results['success'] = false;
                    $results['errors'][] = "{$month}: {$saveResult['error']}";
                } else {
                    $results['total_records'] += $saveResult['records'];
                }
            } else {
                $results['months'][$month] = [
                    'success' => false,
                    'records' => 0,
                    'error' => $fetchResult['error'] ?? 'No data fetched'
                ];
                
                $results['success'] = false;
                $results['errors'][] = "{$month}: " . ($fetchResult['error'] ?? 'No data fetched');
            }
        }

        // Fetch and save OOO data
        $oooFetchResult = $this->fetchOOOData();
        
        if ($oooFetchResult['success'] && !empty($oooFetchResult['data'])) {
            $oooSaveResult = $this->saveOOOData($oooFetchResult['data']);
            
            $results['ooo'] = [
                'success' => $oooSaveResult['success'],
                'records' => $oooSaveResult['records']
            ];
            
            if (!$oooSaveResult['success']) {
                $results['success'] = false;
                $results['errors'][] = "OOO: {$oooSaveResult['error']}";
            } else {
                $results['total_records'] += $oooSaveResult['records'];
            }
        } else {
            $results['ooo'] = [
                'success' => false,
                'records' => 0,
                'error' => $oooFetchResult['error'] ?? 'No data fetched'
            ];
            
            $results['success'] = false;
            $results['errors'][] = "OOO: " . ($oooFetchResult['error'] ?? 'No data fetched');
        }

        $duration = round(microtime(true) - $startTime, 2);
        
        Log::info("HIRBC: Data refresh completed", [
            'duration' => $duration . 's',
            'total_records' => $results['total_records'],
            'success' => $results['success']
        ]);

        return $results;
    }
}