<?php

namespace App\Services\BusinessCentral;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\Regent\OooRegent;
use Carbon\Carbon;
use Exception;

class RegentBusinessCentralService extends BaseApiService
{
    protected $roomEndpoint;
    protected $oooEndpoint;
    protected $businessUnit;

    public function __construct()
    {
        parent::__construct();
        
        $apiConfig = config('regent.api');
        $this->roomEndpoint = $apiConfig['base_url'] . '/' . $apiConfig['company'] . $apiConfig['endpoints']['room_data'];
        $this->oooEndpoint = $apiConfig['base_url'] . '/' . $apiConfig['company'] . $apiConfig['endpoints']['ooo_data'];
        $this->businessUnit = $apiConfig['business_unit'];
    }

    /**
     * Fetch room data for specific month and year
     */
    public function fetchRoomData($month, $year): array
    {
        try {
            Log::info("REGENT: Starting fetch for month {$month} year {$year}");
            
            // Calculate date range
            $startDate = Carbon::create($year, $month, 1)->startOfMonth();
            $endDate = Carbon::create($year, $month, 1)->endOfMonth()->addDay();

            // Build filter (NO QUOTES for dates! Same as HIRBND)
            $filter = sprintf(
                "Business_Unit eq '%s' and Posting_Date ge %s and Posting_Date lt %s",
                $this->businessUnit,
                $startDate->format('Y-m-d'),
                $endDate->format('Y-m-d')
            );

            $endpoint = $this->roomEndpoint . "?\$filter=" . $filter;

            Log::info('REGENT: Fetching with filter', [
                'month' => $month,
                'year' => $year,
                'filter' => $filter,
                'date_range' => $startDate->format('Y-m-d') . ' to ' . $endDate->subDay()->format('Y-m-d')
            ]);

            $startTime = microtime(true);
            
            // ✅ FIX: Get response and extract 'value' key
            $response = $this->get($endpoint);
            $data = $response['value'] ?? $response;
            
            $duration = round(microtime(true) - $startTime, 2);

            Log::info('REGENT: ✅ Fetch completed', [
                'month' => $month,
                'year' => $year,
                'records' => count($data),
                'duration' => "{$duration}s"
            ]);

            return $data;

        } catch (Exception $e) {
            Log::error("REGENT: ❌ Fetch failed for month {$month} year {$year}", [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Fetch OOO data
     */
    public function fetchOOOData(): array
    {
        try {
            // Filter: Business Unit = RBC AND Completed_On = 0001-01-01 (NO QUOTES!)
            $filter = sprintf(
                "Business_Unit eq '%s' and Completed_On eq 0001-01-01",
                $this->businessUnit
            );

            $endpoint = $this->oooEndpoint . "?\$filter=" . $filter;

            Log::info('REGENT: Fetching OOO data', [
                'filter' => $filter,
            ]);

            $startTime = microtime(true);
            
            // ✅ FIX: Get response and extract 'value' key
            $response = $this->get($endpoint);
            $data = $response['value'] ?? $response;
            
            $duration = round(microtime(true) - $startTime, 2);

            Log::info('REGENT: ✅ OOO data fetched', [
                'records' => count($data),
                'duration' => "{$duration}s"
            ]);

            return $data;

        } catch (Exception $e) {
            Log::error('REGENT: ❌ Failed to fetch OOO data', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Save room data to monthly table
     */
    public function saveRoomData($monthName, $data): int
    {
        $tableName = "{$monthName}_regent";
        
        Log::info("REGENT: Saving to {$tableName}", [
            'records' => count($data)
        ]);

        try {
            DB::beginTransaction();

            // ✅ Use delete() instead of truncate()
            DB::table($tableName)->delete();
            Log::info("REGENT: Table truncated", ['table' => $tableName]);

            if (empty($data)) {
                Log::warning("REGENT: No data to save for {$monthName}");
                DB::commit();
                return 0;
            }

            $chunks = array_chunk($data, 500);
            $totalSaved = 0;

            foreach ($chunks as $index => $chunk) {
                $records = array_map(function ($item) {
                    return [
                        'Business_Unit' => $item['Business_Unit'] ?? null,
                        'Room' => $item['Room'] ?? '-',  // ✅ FIX: Default '-' instead of null
                        'RoomType' => $item['RoomType'] ?? null,
                        'Posting_Date' => $item['Posting_Date'] ?? null,
                        'Guest_Full_Name' => $item['Guest_Full_Name'] ?? null,
                        'Market_Code' => $item['Market_Code'] ?? null,
                        'TRX_Code' => $item['TRX_Code'] ?? null,
                        'TRX_Desc' => $item['TRX_Desc'] ?? null,
                        'Amount_1' => $item['Amount_1'] ?? 0,  // ✅ FIX: Default 0 instead of null
                        'Interval' => $item['Interval'] ?? null,
                        'No_Asset' => $item['No_Asset'] ?? null,
                        'Row_No' => $item['Row_No'] ?? null,
                        'odata_etag' => $item['@odata.etag'] ?? null,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }, $chunk);

                DB::table($tableName)->insert($records);
                $totalSaved += count($records);

                Log::info("REGENT: Chunk saved", [
                    'month' => $monthName,
                    'chunk' => $index + 1,
                    'records' => count($records),
                    'total_so_far' => $totalSaved
                ]);
            }

            DB::commit();

            Log::info("REGENT: ✅ {$monthName} data saved successfully", [
                'records' => $totalSaved
            ]);

            return $totalSaved;

        } catch (Exception $e) {
            DB::rollBack();
            
            Log::error("REGENT: ❌ Save failed for {$monthName}", [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
 * Save OOO data
 */
public function saveOOOData($data): int
{
    Log::info('REGENT: Saving OOO data', [
        'records' => count($data)
    ]);

    try {
        DB::beginTransaction();

        // ✅ Use delete() instead of truncate()
        DB::table('ooo_regent')->delete();
        Log::info('REGENT: OOO table truncated');

        if (empty($data)) {
            Log::warning('REGENT: No OOO data to save');
            DB::commit();
            return 0;
        }

        $chunks = array_chunk($data, 100);
        $totalSaved = 0;

        foreach ($chunks as $index => $chunk) {
            $records = array_map(function ($item) {
                return [
                    'odata_etag' => $item['@odata.etag'] ?? null,
                    'Business_Unit' => $item['Business_Unit'] ?? null,
                    'AuxiliaryIndex1' => $item['AuxiliaryIndex1'] ?? null,
                    'Room_No' => $item['Room_No'] ?? '-',
                    'Room_Class' => $item['Room_Class'] ?? null,
                    'Room_Status' => $item['Room_Status'] ?? null, // ✅ ADDED!
                    'Begin_Date' => $item['Begin_Date'] ?? null,
                    'Completed_On' => $item['Completed_On'] ?? null,
                    'End_Date' => $item['End_Date'] ?? null, // ✅ ADDED!
                    'Label' => $item['Label'] ?? null, // ✅ ADDED!
                    'Reason_Code' => $item['Reason_Code'] ?? null, // ✅ ADDED!
                    'Reason_Desc' => $item['Reason_Desc'] ?? null,
                    'Repair_Remark' => $item['Repair_Remark'] ?? null,
                    'Return_Status' => $item['Return_Status'] ?? null, // ✅ ADDED!
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }, $chunk);

            DB::table('ooo_regent')->insert($records);
            $totalSaved += count($records);

            Log::info('REGENT: OOO chunk saved', [
                'chunk' => $index + 1,
                'records' => count($records),
                'total_so_far' => $totalSaved
            ]);
        }

        DB::commit();

        Log::info('REGENT: ✅ OOO data saved successfully', [
            'records' => $totalSaved
        ]);

        return $totalSaved;

    } catch (Exception $e) {
        DB::rollBack();
        
        Log::error('REGENT: ❌ OOO save failed', [
            'error' => $e->getMessage()
        ]);
        throw $e;
    }
}
}