@extends('layouts.regent')

@section('title', 'Revenue Report')

@push('styles')
<style>
    /* ===== STICKY HEADER WITH FILTER & STATS ===== */
    .report-sticky-header {
        position: sticky;
        top: 70px;
        z-index: 100;
        background: white;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        border-radius: 12px;
        margin-bottom: 24px;
    }

    .header-content {
        background: linear-gradient(135deg, #00897b, #26a69a);
        color: white;
        border-radius: 12px 12px 0 0;
        padding: 24px 32px;
    }

    .header-top {
        display: flex;
        justify-content: space-between;
        align-items: start;
        margin-bottom: 20px;
    }

    .header-title h2 {
        font-size: 28px;
        margin-bottom: 4px;
        font-weight: 700;
    }

    .header-title p {
        font-size: 14px;
        opacity: 0.95;
    }

    .header-filter {
        display: flex;
        gap: 12px;
        align-items: center;
        background: rgba(255, 255, 255, 0.15);
        padding: 12px 16px;
        border-radius: 8px;
        backdrop-filter: blur(10px);
    }

    .header-filter label {
        font-size: 13px;
        font-weight: 600;
        margin-right: 8px;
        white-space: nowrap;
    }

    .header-filter select {
        padding: 8px 16px;
        border: 2px solid rgba(255, 255, 255, 0.3);
        border-radius: 6px;
        font-size: 13px;
        background: white;
        color: #333;
        min-width: 150px;
        font-weight: 600;
    }

    .header-filter select:focus {
        outline: none;
        border-color: white;
    }

    .btn-header {
        padding: 8px 20px;
        border: none;
        border-radius: 6px;
        font-weight: 600;
        cursor: pointer;
        font-size: 13px;
        transition: all 0.3s;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .btn-export {
        background: #28a745;
        color: white;
    }

    .btn-export:hover {
        background: #218838;
        transform: translateY(-2px);
    }

    /* ===== STICKY STATS BAR ===== */
    .stats-bar {
        display: grid;
        grid-template-columns: repeat(4, 1fr);
        gap: 1px;
        background: #e0e0e0;
        border-radius: 0 0 12px 12px;
        overflow: hidden;
    }

    .stat-item {
        background: white;
        padding: 16px 20px;
        text-align: center;
    }

    .stat-item:first-child {
        border-radius: 0 0 0 12px;
    }

    .stat-item:last-child {
        border-radius: 0 0 12px 0;
    }

    .stat-label {
        font-size: 11px;
        color: #6c757d;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 6px;
        font-weight: 600;
    }

    .stat-value {
        font-size: 24px;
        font-weight: 700;
        color: #00897b;
    }

    /* ===== REVENUE TABLE WITH STICKY HEADER ===== */
    .revenue-container {
        background: white;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        overflow: hidden;
    }

    .revenue-scroll-wrapper {
        overflow: auto;
        max-height: calc(100vh - 400px);
        position: relative;
    }

    .revenue-table {
        width: 100%;
        border-collapse: collapse;
        font-size: 13px;
        min-width: 1200px;
    }

    .revenue-table thead {
        position: sticky;
        top: 0;
        z-index: 10;
    }

    .revenue-table th {
        background: #00897b;
        color: white;
        padding: 12px 8px;
        text-align: center;
        font-weight: 600;
        border: 1px solid #00695c;
        white-space: nowrap;
    }

    .revenue-table th:not(:first-child) {
        width: 100px;
        min-width: 100px;
        max-width: 100px;
    }

    .revenue-table th:first-child {
        position: sticky;
        left: 0;
        z-index: 11;
        min-width: 120px;
        max-width: 120px;
        width: 120px;
        text-align: left;
        padding-left: 20px;
    }

    .revenue-table td {
        padding: 12px 8px;
        text-align: center;
        border: 1px solid #e0e0e0;
    }

    .revenue-table tbody tr:hover {
        background: #f8f9fa;
    }

    .revenue-table td.room-cell {
        position: sticky;
        left: 0;
        background: #f8f9fa;
        font-weight: 600;
        text-align: left;
        padding-left: 20px;
        z-index: 9;
        border-right: 2px solid #00897b;
        width: 120px;
        min-width: 120px;
        max-width: 120px;
    }

    .revenue-table tbody tr:hover td.room-cell {
        background: #e0f2f1;
    }

    .revenue-table td.amount {
        text-align: right;
        padding-right: 16px;
        font-weight: 600;
        color: #00897b;
    }

    .revenue-table td.total-row {
        background: #b2dfdb;
        font-weight: 700;
        color: #004d40;
    }

    .revenue-table td.average-row {
        background: #e0f2f1;
        font-weight: 700;
        color: #00695c;
    }

    /* ===== EMPTY STATE ===== */
    .empty-state {
        text-align: center;
        padding: 80px 20px;
        color: #6c757d;
    }

    .empty-state i {
        font-size: 64px;
        margin-bottom: 20px;
        color: #00897b;
    }

    .empty-state h3 {
        font-size: 20px;
        margin-bottom: 8px;
        color: #495057;
    }

    .empty-state p {
        font-size: 14px;
    }
</style>
@endpush

@section('content')
    @php
        // Process revenue data
        if($month === 'all') {
            $rooms = [];
            $monthlyRevenue = [];
            
            foreach($revenueData as $monthName => $data) {
                foreach($data as $record) {
                    $rooms[$record->Room] = true;
                    if(!isset($monthlyRevenue[$record->Room][$monthName])) {
                        $monthlyRevenue[$record->Room][$monthName] = 0;
                    }
                    $monthlyRevenue[$record->Room][$monthName] += $record->Amount_1;
                }
            }
            
            $rooms = array_keys($rooms);
            sort($rooms);
            
            $monthlyTotals = [];
            foreach(['january', 'february', 'march', 'april', 'may', 'june', 'july', 'august', 'september', 'october', 'november', 'december'] as $monthName) {
                $monthlyTotals[$monthName] = 0;
                foreach($rooms as $room) {
                    $monthlyTotals[$monthName] += $monthlyRevenue[$room][$monthName] ?? 0;
                }
            }
            
            $grandTotal = array_sum($monthlyTotals);
            
        } else {
            $rooms = [];
            $roomRevenue = [];
            $roomDays = [];
            
            foreach($revenueData as $record) {
                $rooms[$record->Room] = true;
                if(!isset($roomRevenue[$record->Room])) {
                    $roomRevenue[$record->Room] = 0;
                    $roomDays[$record->Room] = [];
                }
                $roomRevenue[$record->Room] += $record->Amount_1;
                $day = date('Y-m-d', strtotime($record->Posting_Date));
                $roomDays[$record->Room][$day] = true;
            }
            
            $rooms = array_keys($rooms);
            sort($rooms);
            
            $totalRevenue = array_sum($roomRevenue);
        }
    @endphp

    <!-- STICKY HEADER: Title + Filter + Stats -->
    <div class="report-sticky-header">
        <!-- Header Content with Filter -->
        <div class="header-content">
            <div class="header-top">
                <div class="header-title">
                    <h2><i class="fas fa-chart-line"></i> Revenue Report</h2>
                    <p>Analyze revenue performance by room and month</p>
                </div>

                <form method="GET" action="{{ route('regent.reports.revenue') }}" class="header-filter">
                    <label for="monthFilter"><i class="fas fa-calendar-alt"></i> Period:</label>
                    <select id="monthFilter" name="month" onchange="this.form.submit()">
                        <option value="all" {{ $month === 'all' ? 'selected' : '' }}>All Months</option>
                        <option value="january" {{ $month === 'january' ? 'selected' : '' }}>January</option>
                        <option value="february" {{ $month === 'february' ? 'selected' : '' }}>February</option>
                        <option value="march" {{ $month === 'march' ? 'selected' : '' }}>March</option>
                        <option value="april" {{ $month === 'april' ? 'selected' : '' }}>April</option>
                        <option value="may" {{ $month === 'may' ? 'selected' : '' }}>May</option>
                        <option value="june" {{ $month === 'june' ? 'selected' : '' }}>June</option>
                        <option value="july" {{ $month === 'july' ? 'selected' : '' }}>July</option>
                        <option value="august" {{ $month === 'august' ? 'selected' : '' }}>August</option>
                        <option value="september" {{ $month === 'september' ? 'selected' : '' }}>September</option>
                        <option value="october" {{ $month === 'october' ? 'selected' : '' }}>October</option>
                        <option value="november" {{ $month === 'november' ? 'selected' : '' }}>November</option>
                        <option value="december" {{ $month === 'december' ? 'selected' : '' }}>December</option>
                    </select>

                    <button type="button" onclick="exportToExcel()" class="btn-header btn-export">
                        <i class="fas fa-file-excel"></i> Export
                    </button>
                </form>
            </div>
        </div>

        <!-- Stats Bar -->
        <div class="stats-bar">
            <div class="stat-item">
                <div class="stat-label"><i class="fas fa-dollar-sign"></i> Total Revenue</div>
                <div class="stat-value" style="font-size: 18px;">Rp {{ number_format($month === 'all' ? $grandTotal : $totalRevenue, 0, ',', '.') }}</div>
            </div>

            <div class="stat-item">
                <div class="stat-label"><i class="fas fa-chart-bar"></i> Average/Room</div>
                <div class="stat-value" style="font-size: 18px;">Rp {{ count($rooms) > 0 ? number_format(($month === 'all' ? $grandTotal : $totalRevenue) / count($rooms), 0, ',', '.') : 0 }}</div>
            </div>

            <div class="stat-item">
                <div class="stat-label"><i class="fas fa-bed"></i> Rooms</div>
                <div class="stat-value">{{ count($rooms) }}</div>
            </div>

            <div class="stat-item">
                <div class="stat-label"><i class="fas fa-calendar-check"></i> Period</div>
                <div class="stat-value" style="font-size: 16px;">{{ $month === 'all' ? '12 Months' : ucfirst($month) }}</div>
            </div>
        </div>
    </div>

    <!-- REVENUE TABLE -->
    <div class="revenue-container">
        @if(count($rooms) === 0)
            <div class="empty-state">
                <i class="fas fa-inbox"></i>
                <h3>No Revenue Data</h3>
                <p>No revenue data found for this period.</p>
            </div>
        @else
            <div class="revenue-scroll-wrapper">
                @if($month === 'all')
                    {{-- All Months View --}}
                    <table class="revenue-table">
                        <thead>
                            <tr>
                                <th>Room</th>
                                @foreach(['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'] as $monthName)
                                    <th>{{ $monthName }}</th>
                                @endforeach
                                <th>Total</th>
                                <th>Average</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($rooms as $room)
                                <tr>
                                    <td class="room-cell">Room {{ $room }}</td>
                                    @php
                                        $roomTotal = 0;
                                        $roomCount = 0;
                                    @endphp
                                    @foreach(['january', 'february', 'march', 'april', 'may', 'june', 'july', 'august', 'september', 'october', 'november', 'december'] as $monthName)
                                        @php
                                            $amount = $monthlyRevenue[$room][$monthName] ?? 0;
                                            $roomTotal += $amount;
                                            if($amount > 0) $roomCount++;
                                        @endphp
                                        <td class="amount">Rp {{ number_format($amount, 0, ',', '.') }}</td>
                                    @endforeach
                                    <td class="amount">Rp {{ number_format($roomTotal, 0, ',', '.') }}</td>
                                    <td class="amount">Rp {{ $roomCount > 0 ? number_format($roomTotal / $roomCount, 0, ',', '.') : 0 }}</td>
                                </tr>
                            @endforeach
                            
                            {{-- Total Row --}}
                            <tr>
                                <td class="total-row" style="text-align: left; padding-left: 20px;">TOTAL</td>
                                @foreach(['january', 'february', 'march', 'april', 'may', 'june', 'july', 'august', 'september', 'october', 'november', 'december'] as $monthName)
                                    <td class="total-row">Rp {{ number_format($monthlyTotals[$monthName], 0, ',', '.') }}</td>
                                @endforeach
                                <td class="total-row">Rp {{ number_format($grandTotal, 0, ',', '.') }}</td>
                                <td class="total-row">-</td>
                            </tr>
                            
                            {{-- Average Row --}}
                            <tr>
                                <td class="average-row" style="text-align: left; padding-left: 20px;">AVERAGE</td>
                                @foreach(['january', 'february', 'march', 'april', 'may', 'june', 'july', 'august', 'september', 'october', 'november', 'december'] as $monthName)
                                    <td class="average-row">Rp {{ count($rooms) > 0 ? number_format($monthlyTotals[$monthName] / count($rooms), 0, ',', '.') : 0 }}</td>
                                @endforeach
                                <td class="average-row">Rp {{ count($rooms) > 0 ? number_format($grandTotal / count($rooms), 0, ',', '.') : 0 }}</td>
                                <td class="average-row">-</td>
                            </tr>
                        </tbody>
                    </table>
                @else
                    {{-- Single Month View --}}
                    <table class="revenue-table">
                        <thead>
                            <tr>
                                <th>Room</th>
                                <th>Revenue</th>
                                <th>Days Occupied</th>
                                <th>ADR</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($rooms as $room)
                                @php
                                    $revenue = $roomRevenue[$room];
                                    $days = count($roomDays[$room]);
                                    $adr = $days > 0 ? $revenue / $days : 0;
                                @endphp
                                <tr>
                                    <td class="room-cell">Room {{ $room }}</td>
                                    <td class="amount">Rp {{ number_format($revenue, 0, ',', '.') }}</td>
                                    <td style="text-align: center;">{{ $days }} days</td>
                                    <td class="amount">Rp {{ number_format($adr, 0, ',', '.') }}</td>
                                </tr>
                            @endforeach
                            
                            {{-- Total Row --}}
                            <tr>
                                <td class="total-row" style="text-align: left; padding-left: 20px;">TOTAL</td>
                                <td class="total-row">Rp {{ number_format($totalRevenue, 0, ',', '.') }}</td>
                                <td class="total-row" colspan="2">-</td>
                            </tr>
                        </tbody>
                    </table>
                @endif
            </div>
        @endif
    </div>
@endsection

@push('scripts')
<script>
    function exportToExcel() {
        alert('Excel export feature coming soon!');
    }
</script>
@endpush
