<?php

namespace App\Http\Controllers;

use App\Models\ItemRequest;
use App\Models\ApprovalFlowTemplate;
use App\Models\User;
use App\Models\Company;
use App\Models\UserRequest;
use App\Models\ApprovalStepTemplate;
use App\Models\ApprovalStepApprover;
use App\Models\ApprovalInstance;
use App\Models\OpnameSchedule;
use App\Models\ItemBeforeS1;
use App\Services\BusinessCentralService;
use App\Models\Location;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use App\Models\ApprovalAction;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Support\TimeFormatter;


class DashboardController extends Controller
{
    // ==========================
    // DASHBOARD VIEW
    // ==========================
    public function index()
    {
        $user = auth()->user();
        $isAdmin = $user->role_id === 1;

        $query = OpnameSchedule::whereIn('status', ['Pending', 'Start', 'Pause'])
            ->orderBy('location')
            ->orderBy('date');

        if (!$isAdmin && $user->business_unit) {
            $query->where('location', $user->business_unit);
        }

        $opnames = $query->get();

        /** ⛔ Hapus return disini */

        /** 🔥 Cek lokasi hanya jika ada jadwal */
        if ($opnames->isNotEmpty()) {
            foreach ($opnames as $opname) {

                $requiredLocations = Location::where('location', $opname->location)
                    ->pluck('code')
                    ->toArray();
                
                $completedLocations = ItemBeforeS1::where('periode', $opname->periode)
                    ->whereIn('location', $requiredLocations)
                    ->where('step', 'step 1')
                    ->distinct()
                    ->pluck('location')
                    ->toArray();

                $opname->location_status = collect($requiredLocations)
                    ->mapWithKeys(fn($loc) => [$loc => in_array($loc, $completedLocations)]);

                $opname->is_done_get_data = (count($completedLocations) === count($requiredLocations));
            }
        }

        return view('dashboard.index', [
            'opnames' => $opnames,
            'isAdmin' => $isAdmin
        ]);
    }




    // ==========================
    // CREATE SCHEDULE
    // ==========================
   public function store(Request $request)
    {
        $validated = $request->validate([
            'periode'  => ['required', 'regex:/^\d{4}-\d{2}$/'],
            'start_at' => ['required', 'date'],
            'end_at'   => ['required', 'date', 'after_or_equal:start_at'],
        ]);

        // ⏱ Samakan jam (ambil jam sekarang)
        $startAt = Carbon::parse($validated['start_at'])
            ->setTimeFrom(now());

        $endAt = Carbon::parse($validated['end_at'])
            ->setTimeFrom($startAt);

        $groups = ['CI', 'HIC', 'RBC'];

        foreach ($groups as $loc) {
            OpnameSchedule::create([
                'date'       => now(),
                'periode'    => $validated['periode'],
                'location'   => $loc,
                'start_at'   => $startAt,
                'end_at'     => $endAt,
                'status'     => 'Pending',
                'remaining_minutes' => null,
                'created_by' => auth()->id(),
            ]);
        }

        // 🔑 PENTING: RESPONSE SESUAI JENIS REQUEST
        if ($request->ajax()) {
            return response()->json([
                'status'   => 'success',
                'message'  => '3 Jadwal Stock Opname berhasil dibuat berdasarkan lokasi CI, HIC & RBC',
                'redirect' => route('dashboard'),
            ]);
        }

        return redirect()
            ->route('dashboard')
            ->with('success', '3 Jadwal Stock Opname berhasil dibuat berdasarkan lokasi CI, HIC & RBC');
    }


    // ==========================
    // START
    // ==========================

    public function start($id)
    {
        $opname = OpnameSchedule::findOrFail($id);
        $now = now();

        // ======================================
        // 1. START PERTAMA: CEK TANGGAL
        // ======================================
        if ($opname->status === 'Pending' && $now->lt($opname->start_at)) {
            return back()->with(
                'error',
                'Stock opname belum bisa dimulai sebelum ' .
                $opname->start_at->format('d M Y ')
            );
        }

        // ======================================
        // 2. TENTUKAN DURASI (MENIT)
        // ======================================
        if ($opname->status === 'Pause' && $opname->remaining_minutes !== null) {
            // 🔁 RESUME
            $minutes = $opname->remaining_minutes;
        } else {
            // ▶ START PERTAMA
            $minutes = max(
                0,
                $opname->start_at->diffInMinutes($opname->end_at)
            );
        }

        if ($minutes <= 0) {
            return back()->with('error', 'Durasi schedule sudah habis.');
        }

        // ======================================
        // 3. RESET TIMER BERDASARKAN NOW
        // ======================================
        $opname->update([
            'status' => 'Start',
            'start_at' => $now,
            'end_at' => $now->copy()->addMinutes($minutes),
            'remaining_minutes' => null,
        ]);

        return back()->with(
            'success',
            "Stock Opname {$opname->location} dimulai (sisa " .
            TimeFormatter::humanMinutes($minutes) .
            ")"
        ); 
          
    }

    // ==========================
    // PAUSE
    // ==========================
    public function pause($id)
    {
        $opname = OpnameSchedule::findOrFail($id);

        if ($opname->status !== 'Start') {
            return back()->with('error', 'Schedule tidak sedang berjalan.');
        }

        $now = now();

        // ======================================
        // HITUNG SISA MENIT
        // ======================================
        $remaining = max(
            0,
            $now->diffInMinutes($opname->end_at, false)
        );

        $opname->update([
            'status' => 'Pause',
            'remaining_minutes' => $remaining,
        ]);

        return back()->with(
            'success',
            "Stock Opname {$opname->location} dijeda (sisa " .
            TimeFormatter::humanMinutes($remaining) .
            ")"
        );
    }

    // ==========================
    // EXTEND
    // ==========================
    public function extend(Request $request, $id)
    {
        $validated = $request->validate([
            'days'    => 'nullable|integer|min:0',
            'hours'   => 'nullable|integer|min:0',
            'minutes' => 'nullable|integer|min:0',
        ]);

        $total = ($validated['days'] ?? 0) * 1440 
            + ($validated['hours'] ?? 0) * 60 
            + ($validated['minutes'] ?? 0);

        if ($total <= 0) {
            return $request->ajax()
                ? response()->json(['status' => 'error', 'message' => 'Durasi extend tidak valid'], 422)
                : redirect()->route('dashboard')->with('error', 'Durasi extend tidak valid');
        }

        $opname = OpnameSchedule::findOrFail($id);
        $opname->update([
            'end_at' => \Carbon\Carbon::parse($opname->end_at)->addMinutes($total)
        ]);

        // 🔥 JIKA AJAX (fetch)
        if ($request->ajax()) {
            return response()->json([
                'status'  => 'success',
                'message' => "Waktu opname lokasi {$opname->location} ditambah",
                'new_end' => $opname->end_at->format('Y-m-d H:i:s')
            ]);
        }

        // 🔥 JIKA FORM SUBMIT BIASA
        return redirect()->route('dashboard')
            ->with('success', "Waktu opname lokasi {$opname->location} berhasil di-extend!");
    }


    // ==========================
    // FINISH
    // ==========================
    public function finish($id)
    {
        $opname = OpnameSchedule::findOrFail($id);

        $opname->update([
            'status'   => 'End',
            'start_at' => now(),
            'end_at'   => now(),
        ]);

        return back()->with('success', "Stock Opname lokasi {$opname->location} selesai");
    }

    // ==========================
    // CANCEL
    // ==========================
    public function cancel($id)
    {
        $opname = OpnameSchedule::findOrFail($id);
        $opname->update(['status' => 'Cancel']);

        return back()->with('success', "Stock Opname lokasi {$opname->location} dibatalkan");
    }


    public function processSingleLocationStep1($id, $loc, BusinessCentralService $bc)
    {
        Log::info("STEP 1 | Proses lokasi {$loc} untuk opname ID {$id}");

        try {
            $schedule = OpnameSchedule::findOrFail($id);
            $periode  = $schedule->periode ?? date('Y-m');

            // 🔹 ambil data BC (kumulatif sampai latest)
            $data = $bc->getStockOpnameItems($loc);

            if (isset($data['error'])) {
                throw new \Exception("BC Error: {$data['error']}");
            }

            if (!is_array($data) || empty($data)) {
                throw new \Exception("Data BC kosong untuk lokasi {$loc}");
            }

            foreach ($data as $item) {

                ItemBeforeS1::updateOrCreate(
                    [
                        'location' => $item['LocationCode'] ?? $loc,
                        'periode'  => $periode,
                        'item_no'  => $item['ItemNo'],
                    ],
                    [
                        'step' => 'step 1',
                        'item_description'  => $item['Description'] ?? null,
                        'uom'               => $item['UoMCode'] ?? null,
                        'unit_cost'         => $item['UnitCost'] ?? 0,

                        // 🔥 snapshot posisi saat step 1
                        'opening_balance'   => $item['Qty'] ?? 0,

                        // 🔥 TOTAL KUMULATIF dari BC
                        'last_qty_increase' => $item['QtyIncrease'] ?? 0,
                        'last_qty_decrease' => $item['QtyDecrease'] ?? 0,

                        'remarks'           => 'STEP 1 – baseline',
                    ]
                );
            }

            return response()->json([
                'status'  => 'success',
                'message' => "STEP 1 berhasil untuk lokasi {$loc}",
                'count'   => count($data),
            ]);

        } catch (\Throwable $e) {
            Log::error("STEP 1 ERROR | {$loc}", ['error' => $e->getMessage()]);

            return response()->json([
                'status'  => 'error',
                'message' => $e->getMessage(),
            ], 500);
        }
    }


    public function rollbackOpnameStep1($id)
    {
        $schedule = OpnameSchedule::findOrFail($id);

        $periode       = $schedule->periode;
        $locationGroup = $schedule->location; // contoh: CI, RBC, HIC

        ItemBeforeS1::where('periode', $periode)
            ->where('location', 'like', $locationGroup . '%')
            ->delete();

        return response()->json([
            'status'  => 'success',
            'message' => "Rollback STEP 1 selesai untuk {$locationGroup} periode {$periode}",
        ]);
    }

    public function updateDate(Request $request, $id)
    {
        $request->validate([
            'start_at' => 'required|date',
            'end_at'   => 'required|date|after_or_equal:start_at',
        ]);

        $opname = OpnameSchedule::findOrFail($id);

        $opname->update([
            'start_at' => $request->start_at,
            'end_at'   => $request->end_at,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Tanggal schedule berhasil diperbarui.'
        ]);
    }

}
