<?php
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'borrower') {
    echo '<div class="text-center py-4 text-red-500">Access denied</div>';
    exit();
}

include '../config/connection.php';

$transaction_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$borrower_id = $_SESSION['user_id'];

// Get transaction detail
$query = "SELECT st.*, bu.nama_borrower, bu.nik
          FROM stock_transactions st
          LEFT JOIN borrower_users bu ON st.borrower_id = bu.id
          WHERE st.id = ? AND st.borrower_id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("ii", $transaction_id, $borrower_id);
$stmt->execute();
$transaction = $stmt->get_result()->fetch_assoc();

if (!$transaction) {
    echo '<div class="text-center py-4 text-red-500">Transaction not found</div>';
    exit();
}

// Get transaction items
$items_query = "SELECT ti.*, csi.code_barang_lemari5, csi.lokasi_penyimpanan, csi.lemari_number,
                       mc.color_name, cc.component_name
                FROM transaction_items ti
                LEFT JOIN color_stock_items csi ON ti.stock_item_id = csi.id
                LEFT JOIN master_color mc ON csi.master_color_id = mc.id
                LEFT JOIN color_components cc ON csi.component_id = cc.id
                WHERE ti.transaction_id = ?
                ORDER BY ti.code_color";
$items_stmt = $conn->prepare($items_query);
$items_stmt->bind_param("i", $transaction_id);
$items_stmt->execute();
$items_result = $items_stmt->get_result();
?>

<div class="space-y-6">
    <!-- Transaction Info -->
    <div class="bg-gray-50 rounded-lg p-4">
        <h4 class="font-semibold text-gray-800 mb-3">Informasi Transaksi</h4>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
            <div>
                <span class="font-medium text-gray-600">No. Transaksi:</span>
                <p class="font-mono text-purple-600"><?php echo htmlspecialchars($transaction['transaction_number']); ?></p>
            </div>
            <div>
                <span class="font-medium text-gray-600">Status:</span>
                <?php
                $status_class = '';
                $status_text = '';
                switch($transaction['status']) {
                    case 'approved':
                        $status_class = 'bg-green-100 text-green-800';
                        $status_text = 'Approved';
                        break;
                    case 'returned':
                        $status_class = 'bg-blue-100 text-blue-800';
                        $status_text = 'Returned';
                        break;
                    case 'overdue':
                        $status_class = 'bg-red-100 text-red-800';
                        $status_text = 'Overdue';
                        break;
                    default:
                        $status_class = 'bg-gray-100 text-gray-800';
                        $status_text = ucfirst($transaction['status']);
                }
                ?>
                <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $status_class; ?>">
                    <?php echo $status_text; ?>
                </span>
            </div>
            <div>
                <span class="font-medium text-gray-600">Tanggal Pinjam:</span>
                <p><?php echo date('d M Y H:i', strtotime($transaction['transaction_date'])); ?></p>
            </div>
            <div>
                <span class="font-medium text-gray-600">Tanggal Kembali:</span>
                <p><?php echo date('d M Y H:i', strtotime($transaction['expected_return_date'])); ?></p>
            </div>
            <div>
                <span class="font-medium text-gray-600">Peminjam:</span>
                <p><?php echo htmlspecialchars($transaction['nama_borrower']); ?></p>
            </div>
            <div>
                <span class="font-medium text-gray-600">NIK:</span>
                <p><?php echo htmlspecialchars($transaction['nik']); ?></p>
            </div>
        </div>
    </div>

    <!-- Items List -->
    <div>
        <h4 class="font-semibold text-gray-800 mb-3">Daftar Item Dipinjam</h4>
        <div class="overflow-x-auto">
            <table class="min-w-full bg-white border border-gray-200 rounded-lg">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Code Color</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Color Name</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Component</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Lokasi</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                        <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Aksi</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-200">
                    <?php if ($items_result->num_rows > 0): ?>
                        <?php while ($item = $items_result->fetch_assoc()): ?>
                        <tr class="hover:bg-gray-50">
                            <td class="px-4 py-3 text-sm font-medium text-purple-600">
                                <?php echo htmlspecialchars($item['code_color']); ?>
                            </td>
                            <td class="px-4 py-3 text-sm text-gray-900">
                                <?php echo htmlspecialchars($item['color_name'] ?? '-'); ?>
                            </td>
                            <td class="px-4 py-3 text-sm text-gray-500">
                                <?php echo htmlspecialchars($item['component_name'] ?? '-'); ?>
                            </td>
                            <td class="px-4 py-3 text-sm text-gray-500">
                                <div>
                                    <div><?php echo htmlspecialchars($item['lokasi_penyimpanan'] ?? '-'); ?></div>
                                    <div class="text-xs">Lemari: <?php echo htmlspecialchars($item['lemari_number'] ?? '-'); ?></div>
                                </div>
                            </td>
                            <td class="px-4 py-3 text-sm">
                                <?php
                                $borrow_status_class = '';
                                $borrow_status_text = '';
                                switch($item['borrow_status']) {
                                    case 'borrowed':
                                        $borrow_status_class = 'bg-yellow-100 text-yellow-800';
                                        $borrow_status_text = 'Borrowed';
                                        break;
                                    case 'returned':
                                        $borrow_status_class = 'bg-green-100 text-green-800';
                                        $borrow_status_text = 'Returned';
                                        break;
                                    default:
                                        $borrow_status_class = 'bg-gray-100 text-gray-800';
                                        $borrow_status_text = ucfirst($item['borrow_status'] ?? 'Unknown');
                                }
                                ?>
                                <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full <?php echo $borrow_status_class; ?>">
                                    <?php echo $borrow_status_text; ?>
                                </span>
                            </td>
                            <td class="px-4 py-3 text-sm">
                                <?php if ($item['borrow_status'] === 'borrowed'): ?>
                                <button onclick="openReturnScanModal(<?php echo $item['id']; ?>, '<?php echo htmlspecialchars($item['code_color']); ?>', '<?php echo htmlspecialchars($item['color_name'] ?? '-'); ?>')" 
                                        class="bg-green-500 hover:bg-green-600 text-white px-3 py-1 rounded text-xs transition-colors">
                                    <i class="fas fa-qrcode mr-1"></i>Kembalikan
                                </button>
                                <?php elseif ($item['borrow_status'] === 'returned'): ?>
                                <span class="text-green-600 text-xs">
                                    <i class="fas fa-check-circle mr-1"></i>Sudah Dikembalikan
                                </span>
                                <?php if ($item['actual_return_date']): ?>
                                <div class="text-xs text-gray-500 mt-1">
                                    <?php echo date('d M Y H:i', strtotime($item['actual_return_date'])); ?>
                                </div>
                                <?php endif; ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" class="px-4 py-8 text-center text-gray-500">Tidak ada item ditemukan</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Timeline (if there are notes or updates) -->
    <?php if ($transaction['notes']): ?>
    <div>
        <h4 class="font-semibold text-gray-800 mb-3">Catatan</h4>
        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <p class="text-sm text-yellow-800"><?php echo nl2br(htmlspecialchars($transaction['notes'])); ?></p>
        </div>
    </div>
    <?php endif; ?>

    <!-- Summary -->
    <div class="bg-purple-50 border border-purple-200 rounded-lg p-4">
        <h4 class="font-semibold text-purple-800 mb-2">Ringkasan</h4>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
            <div>
                <span class="font-medium text-purple-600">Total Item:</span>
                <?php 
                $items_result->data_seek(0); // Reset pointer
                $total_items = $items_result->num_rows;
                ?>
                <p class="text-purple-800 font-bold"><?php echo $total_items; ?> item(s)</p>
            </div>
            <div>
                <span class="font-medium text-purple-600">Durasi Pinjam:</span>
                <?php
                $start = new DateTime($transaction['transaction_date']);
                $end = new DateTime($transaction['expected_return_date']);
                $duration = $start->diff($end)->days;
                ?>
                <p class="text-purple-800 font-bold"><?php echo $duration; ?> hari</p>
            </div>
            <div>
                <span class="font-medium text-purple-600">Dibuat:</span>
                <p class="text-purple-800 font-bold"><?php echo date('d M Y H:i', strtotime($transaction['created_at'])); ?></p>
            </div>
        </div>
    </div>
</div>