<!-- print_qr.php -->
<?php
// Start session
session_start();

// Include database connection
require_once "../config/connection.php";

// Check if user is logged in and is IT user
if (!is_logged_in() || !is_user_type('it')) {
    header("Location: ../login.php");
    exit();
}

// Check if item_id is provided
if (!isset($_GET['item_id']) || empty($_GET['item_id'])) {
    echo "Item ID is required";
    exit();
}

$item_id = (int)$_GET['item_id'];

try {
    // Get complete item data
    $query = "
        SELECT 
            csi.id,
            csi.kode_barang,
            csi.stock_number,
            csi.status,
            csi.lokasi,
            csi.rack_no,
            csi.qr_code,
            csi.tanggal_pembuatan,
            csi.tanggal_kedaluwarsa,
            csi.keterangan,
            cc.component_name,
            csi.component_id,
            csi.color_id,
            mcp.code_color,
            mcp.deskripsi as color_description,
            mcp.lemari,
            b.nama_buyer,
            b.kode_buyer
        FROM color_stock_items_semarang csi
        JOIN color_components_semarang cc ON csi.component_id = cc.id
        JOIN master_color_panel_semarang mcp ON csi.color_id = mcp.id
        JOIN buyers_semarang b ON mcp.buyer_id = b.id
        WHERE csi.id = ?
    ";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param("i", $item_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo "Item not found";
        exit();
    }
    
    $item = $result->fetch_assoc();
    $stmt->close();
    
    // Check if QR code exists
    if (empty($item['qr_code'])) {
        echo "QR code not found for this item";
        exit();
    }
    
    // Get approval data with signatures from approval_history_semarang
    $approval_query = "
        SELECT 
            ca.id as approval_id,
            ca.ka_unit_signature,
            ca.color_room_signature,
            ca.akzonobel_signature,
            ca.approved_by_signature,
            ca.qc_signature,
            ca.qc_buyer_signature,
            au_ka.nama_lengkap as ka_unit_name,
            au_cr.nama_lengkap as color_room_name,
            au_ak.nama_lengkap as akzonobel_name,
            au_ap.nama_lengkap as approved_by_name,
            au_qc.nama_lengkap as qc_name,
            au_qcb.nama_lengkap as qc_buyer_name
        FROM color_approvals_semarang ca
        LEFT JOIN approval_users_semarang au_ka ON ca.ka_unit_approved_by = au_ka.id
        LEFT JOIN approval_users_semarang au_cr ON ca.color_room_approved_by = au_cr.id
        LEFT JOIN approval_users_semarang au_ak ON ca.akzonobel_approved_by = au_ak.id
        LEFT JOIN approval_users_semarang au_ap ON ca.approved_by_approved_by = au_ap.id
        LEFT JOIN approval_users_semarang au_qc ON ca.qc_approved_by = au_qc.id
        LEFT JOIN approval_users_semarang au_qcb ON ca.qc_buyer_approved_by = au_qcb.id
        WHERE ca.color_id = ? AND ca.component_id = ?
        LIMIT 1
    ";
    
    $stmt = $conn->prepare($approval_query);
    $stmt->bind_param("ii", $item['color_id'], $item['component_id']);
    $stmt->execute();
    $approval_result = $stmt->get_result();
    $approval_data = $approval_result->fetch_assoc();
    $stmt->close();
    
} catch (Exception $e) {
    echo "Database error: " . $e->getMessage();
    exit();
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Print QR Code - <?php echo htmlspecialchars($item['kode_barang']); ?></title>
    <style>
        @page {
            size: A4 portrait;
            margin: 10mm;
        }
        
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background-color: #f5f5f5;
        }
        
        .print-container {
            background: white;
            border: none;
            border-radius: 8px;
            padding: 40px;
            width: 100%;
            max-width: 700px;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
            height: fit-content;
        }
        
        .header {
            margin-bottom: 30px;
        }
        
        .buyer-title {
            font-size: 28px;
            font-weight: bold;
            margin-bottom: 15px;
            color: #333;
        }
        
        .component-title {
            font-size: 22px;
            color: #666;
            margin-bottom: 20px;
        }
        
        .qr-section {
            margin: 30px 0;
        }
        
        .qr-code {
            margin: 15px auto;
            display: block;
        }
        
        .details-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin: 20px 0;
            text-align: left;
            font-size: 18px;
        }
        
        .detail-item {
            margin-bottom: 6px;
            border-bottom: none;
        }
        
        .detail-label {
            font-weight: bold;
            color: #333;
            font-size: 18px;
            border-bottom: none;
        }
        
        .detail-value {
            color: #666;
            margin-top: 4px;
            font-size: 16px;
            border-bottom: none;
        }
        
        .signatures {
            margin-top: 20px;
            border-top: none;
            padding-top: 15px;
        }
        
        .signature-row {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr;
            gap: 40px;
            margin-bottom: 35px;
            font-size: 16px;
        }
        
        .signature-row:last-child {
            margin-bottom: 25px;
        }
        
        .signature-item {
            text-align: center;
            padding: 8px 0;
        }
        
        .signature-label {
            font-size: 16px;
            color: #666;
            font-weight: bold;
            margin-bottom: 10px;
        }
        
        .signature-image {
            max-width: 150px;
            max-height: 60px;
            margin: 5px auto;
            display: block;
        }
        
        .signature-line {
            border-bottom: 1px solid #333;
            margin: 10px 0;
            height: 2px;
        }
        
        .signature-name {
            font-size: 14px;
            color: #888;
            margin-top: 5px;
        }
        
        @media print {
            body {
                background-color: white;
                padding: 0;
                margin: 0;
            }
            
            .print-container {
                box-shadow: none;
                border: none;
                margin: 0;
                max-width: none;
                width: auto;
                height: auto;
                page-break-inside: avoid;
            }
            
            .no-print {
                display: none;
            }
        }
        
        .print-button {
            background-color: #4CAF50;
            color: white;
            padding: 15px 30px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 16px;
            margin-bottom: 30px;
        }
        
        .print-button:hover {
            background-color: #45a049;
        }
    </style>
</head>
<body>
    <div>
        <button class="print-button no-print" onclick="window.print()">
            🖨️ Print QR Code
        </button>
        
        <div class="print-container">
            <!-- Header -->
            <div class="header">
                <div class="buyer-title">
                    <?php echo htmlspecialchars($item['nama_buyer']); ?> - <?php echo htmlspecialchars($item['code_color']); ?>
                </div>
                <div class="component-title">
                    <?php echo htmlspecialchars($item['component_name']); ?> (<?php echo htmlspecialchars($item['kode_barang']); ?>)
                </div>
            </div>
            
            <!-- QR Code Section -->
            <div class="qr-section">
                <img src="<?php echo $item['qr_code']; ?>" alt="QR Code" class="qr-code" style="width: 250px; height: 250px;">
            </div>
            
            <!-- Details Grid -->
            <div class="details-grid">
                <div class="detail-item">
                    <div class="detail-label">Buyer:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($item['nama_buyer']); ?></div>
                </div>
                
                <div class="detail-item">
                    <div class="detail-label">Lokasi:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($item['lokasi']); ?></div>
                </div>
                
                <div class="detail-item">
                    <div class="detail-label">Color Code:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($item['code_color']); ?></div>
                </div>
                
                <div class="detail-item">
                    <div class="detail-label">Rack No:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($item['rack_no']); ?></div>
                </div>
                
                <div class="detail-item">
                    <div class="detail-label">Component:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($item['component_name']); ?></div>
                </div>
                
                <div class="detail-item">
                    <div class="detail-label">Status:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($item['status']); ?></div>
                </div>
                
                <div class="detail-item">
                    <div class="detail-label">Code Barang:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($item['kode_barang']); ?></div>
                </div>
                
                <div class="detail-item">
                    <div class="detail-label">Tgl Pembuatan:</div>
                    <div class="detail-value"><?php echo $item['tanggal_pembuatan'] ? date('d-m-Y', strtotime($item['tanggal_pembuatan'])) : '-'; ?></div>
                </div>
                
                <div class="detail-item">
                    <div class="detail-label">Tgl Kedaluwarsa:</div>
                    <div class="detail-value"><?php echo $item['tanggal_kedaluwarsa'] ? date('d-m-Y', strtotime($item['tanggal_kedaluwarsa'])) : '-'; ?></div>
                </div>
            </div>
            
            <!-- Signatures Section -->
            <div class="signatures">
                <!-- First Row -->
                <div class="signature-row">
                    <div class="signature-item">
                        <div class="signature-label">K A Unit</div>
                        <?php if ($approval_data && !empty($approval_data['ka_unit_signature'])): ?>
                            <img src="<?php echo $approval_data['ka_unit_signature']; ?>" alt="Signature" class="signature-image">
                        <?php endif; ?>
                        <div class="signature-line"></div>
                        <div class="signature-name">
                            <?php echo $approval_data && $approval_data['ka_unit_name'] ? htmlspecialchars($approval_data['ka_unit_name']) : 'Y Andi S'; ?>
                        </div>
                    </div>
                    
                    <div class="signature-item">
                        <div class="signature-label">Color Room</div>
                        <?php if ($approval_data && !empty($approval_data['color_room_signature'])): ?>
                            <img src="<?php echo $approval_data['color_room_signature']; ?>" alt="Signature" class="signature-image">
                        <?php endif; ?>
                        <div class="signature-line"></div>
                        <div class="signature-name">
                            <?php echo $approval_data && $approval_data['color_room_name'] ? htmlspecialchars($approval_data['color_room_name']) : 'Alifi A.U'; ?>
                        </div>
                    </div>
                    
                    <div class="signature-item">
                        <div class="signature-label">Akzonobel</div>
                        <?php if ($approval_data && !empty($approval_data['akzonobel_signature'])): ?>
                            <img src="<?php echo $approval_data['akzonobel_signature']; ?>" alt="Signature" class="signature-image">
                        <?php endif; ?>
                        <div class="signature-line"></div>
                        <div class="signature-name">
                            <?php echo $approval_data && $approval_data['akzonobel_name'] ? htmlspecialchars($approval_data['akzonobel_name']) : 'Asmari'; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Second Row -->
                <div class="signature-row">
                    <div class="signature-item">
                        <div class="signature-label">Approved by</div>
                        <?php if ($approval_data && !empty($approval_data['approved_by_signature'])): ?>
                            <img src="<?php echo $approval_data['approved_by_signature']; ?>" alt="Signature" class="signature-image">
                        <?php endif; ?>
                        <div class="signature-line"></div>
                        <div class="signature-name">
                            <?php echo $approval_data && $approval_data['approved_by_name'] ? htmlspecialchars($approval_data['approved_by_name']) : 'Eris'; ?>
                        </div>
                    </div>
                    
                    <div class="signature-item">
                        <div class="signature-label">Approved by QC</div>
                        <?php if ($approval_data && !empty($approval_data['qc_signature'])): ?>
                            <img src="<?php echo $approval_data['qc_signature']; ?>" alt="Signature" class="signature-image">
                        <?php endif; ?>
                        <div class="signature-line"></div>
                        <div class="signature-name">
                            <?php echo $approval_data && $approval_data['qc_name'] ? htmlspecialchars($approval_data['qc_name']) : 'Suhari yanto'; ?>
                        </div>
                    </div>
                    
                    <div class="signature-item">
                        <div class="signature-label">Approved by QC Buyer</div>
                        <?php if ($approval_data && !empty($approval_data['qc_buyer_signature'])): ?>
                            <img src="<?php echo $approval_data['qc_buyer_signature']; ?>" alt="Signature" class="signature-image">
                        <?php endif; ?>
                        <div class="signature-line"></div>
                        <div class="signature-name">
                            <?php echo $approval_data && $approval_data['qc_buyer_name'] ? htmlspecialchars($approval_data['qc_buyer_name']) : 'QC Buyer'; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Auto print when page loads (optional)
        // window.onload = function() {
        //     window.print();
        // }
        
        // Close window after printing
        window.onafterprint = function() {
            // window.close();
        }
    </script>
</body>
</html>

<?php $conn->close(); ?>