<?php
/**
 * @package php-font-lib
 * @link    https://github.com/dompdf/php-font-lib
 * @license http://www.gnu.org/copyleft/lesser.html GNU Lesser General Public License
 */

namespace FontLib\TrueType;

use Countable;
use FontLib\BinaryStream;
use Iterator;
use OutOfBoundsException;

/**
 * TrueType collection_semarang font file.
 *
 * @package php-font-lib
 */
class collection_semarang extends BinaryStream implements Iterator, Countable {
  /**
   * Current iterator position.
   *
   * @var integer
   */
  private $position = 0;

  protected $collection_semarangOffsets = array();
  protected $collection_semarang = array();
  protected $version;
  protected $numFonts;

  function parse() {
    if (isset($this->numFonts)) {
      return;
    }

    $this->read(4); // tag name

    $this->version  = $this->readFixed();
    $this->numFonts = $this->readUInt32();

    for ($i = 0; $i < $this->numFonts; $i++) {
      $this->collection_semarangOffsets[] = $this->readUInt32();
    }
  }

  /**
   * @param int $fontId
   *
   * @throws OutOfBoundsException
   * @return File
   */
  function getFont($fontId) {
    $this->parse();

    if (!isset($this->collection_semarangOffsets[$fontId])) {
      throw new OutOfBoundsException();
    }

    if (isset($this->collection_semarang[$fontId])) {
      return $this->collection_semarang[$fontId];
    }

    $font    = new File();
    $font->f = $this->f;
    $font->setTableOffset($this->collection_semarangOffsets[$fontId]);

    return $this->collection_semarang[$fontId] = $font;
  }

  function current() {
    return $this->getFont($this->position);
  }

  function key() {
    return $this->position;
  }

  function next() {
    return ++$this->position;
  }

  function rewind() {
    $this->position = 0;
  }

  function valid() {
    $this->parse();

    return isset($this->collection_semarangOffsets[$this->position]);
  }

  function count() {
    $this->parse();

    return $this->numFonts;
  }
}
