<?php
require_once '../config/connection.php';
checkUserType(['borrower']);

// Get borrower stats
$borrower_id = $_SESSION['employee_id'];

// Count pending requests
$stmt = $pdo->prepare("SELECT COUNT(*) as pending FROM borrow_history WHERE borrower_employee_id = ? AND status = 'pending_approval'");
$stmt->execute([$borrower_id]);
$pending_count = $stmt->fetch()['pending'];

// Count active borrows
$stmt = $pdo->prepare("SELECT COUNT(*) as active FROM borrow_history WHERE borrower_employee_id = ? AND status = 'borrowed'");
$stmt->execute([$borrower_id]);
$active_count = $stmt->fetch()['active'];

// Count total history
$stmt = $pdo->prepare("SELECT COUNT(*) as total FROM borrow_history WHERE borrower_employee_id = ?");
$stmt->execute([$borrower_id]);
$total_count = $stmt->fetch()['total'];

// Count returned items
$stmt = $pdo->prepare("SELECT COUNT(*) as returned FROM borrow_history WHERE borrower_employee_id = ? AND status = 'returned'");
$stmt->execute([$borrower_id]);
$returned_count = $stmt->fetch()['returned'];

// Get recent history (last 5)
$stmt = $pdo->prepare("
    SELECT bh.*, a.asset_name, a.asset_number 
    FROM borrow_history bh 
    JOIN assets a ON bh.asset_id = a.id 
    WHERE bh.borrower_employee_id = ? 
    ORDER BY bh.created_at DESC 
    LIMIT 5
");
$stmt->execute([$borrower_id]);
$recent_history = $stmt->fetchAll();

// Get stats by organization
$borrower_org = getBorrowerOrganization($pdo, $borrower_id);
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_org_borrows,
        COUNT(CASE WHEN status = 'borrowed' THEN 1 END) as active_org_borrows
    FROM borrow_history 
    WHERE organization = ?
");
$stmt->execute([$borrower_org]);
$org_stats = $stmt->fetch();

// Get available assets count
$stmt = $pdo->prepare("
    SELECT COUNT(*) as available_count
    FROM assets 
    WHERE status_borrower = 'available' AND kondisi IN ('baik', 'rusak')
");
$stmt->execute();
$available_count = $stmt->fetch()['available_count'];

// GET ALL DATA FOR EACH STATUS - Langsung ambil semua data di awal
// Pending data
$stmt = $pdo->prepare("
    SELECT bh.*, a.asset_name, a.asset_number, a.asset_category 
    FROM borrow_history bh 
    JOIN assets a ON bh.asset_id = a.id 
    WHERE bh.borrower_employee_id = ? AND bh.status = 'pending_approval'
    ORDER BY bh.created_at DESC
");
$stmt->execute([$borrower_id]);
$pending_data = $stmt->fetchAll();

// Borrowed data
$stmt = $pdo->prepare("
    SELECT bh.*, a.asset_name, a.asset_number, a.asset_category 
    FROM borrow_history bh 
    JOIN assets a ON bh.asset_id = a.id 
    WHERE bh.borrower_employee_id = ? AND bh.status = 'borrowed'
    ORDER BY bh.created_at DESC
");
$stmt->execute([$borrower_id]);
$borrowed_data = $stmt->fetchAll();

// Returned data
$stmt = $pdo->prepare("
    SELECT bh.*, a.asset_name, a.asset_number, a.asset_category 
    FROM borrow_history bh 
    JOIN assets a ON bh.asset_id = a.id 
    WHERE bh.borrower_employee_id = ? AND bh.status = 'returned'
    ORDER BY bh.created_at DESC
");
$stmt->execute([$borrower_id]);
$returned_data = $stmt->fetchAll();

// All data
$stmt = $pdo->prepare("
    SELECT bh.*, a.asset_name, a.asset_number, a.asset_category 
    FROM borrow_history bh 
    JOIN assets a ON bh.asset_id = a.id 
    WHERE bh.borrower_employee_id = ?
    ORDER BY bh.created_at DESC
");
$stmt->execute([$borrower_id]);
$all_data = $stmt->fetchAll();

// Function to render table
function renderDataTable($data, $status) {
    if (empty($data)) {
        return '
            <div class="text-center py-12">
                <svg class="w-20 h-20 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 13V6a2 2 0 00-2-2H6a2 2 0 00-2 2v7m16 0v5a2 2 0 01-2 2H6a2 2 0 01-2-2v-5m16 0h-2.586a1 1 0 00-.707.293l-2.414 2.414a1 1 0 01-.707.293h-3.172a1 1 0 01-.707-.293l-2.414-2.414A1 1 0 006.586 13H4"></path>
                </svg>
                <p class="text-gray-500 text-lg font-medium">Tidak ada data</p>
                <p class="text-gray-400 text-sm mt-2">Belum ada peminjaman dengan status ini</p>
            </div>
        ';
    }
    
    $html = '
        <div class="overflow-x-auto -mx-2 sm:mx-0">
            <div class="inline-block min-w-full align-middle">
                <div class="overflow-hidden border border-gray-200 sm:rounded-lg">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th scope="col" class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Asset</th>
                                <th scope="col" class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden md:table-cell">workcenter</th>
                                <th scope="col" class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden lg:table-cell">Preference Pro</th>
                                <th scope="col" class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>';
    
    if ($status === 'returned' || $status === 'all') {
        $html .= '
                                <th scope="col" class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden lg:table-cell">Tanggal Kembali</th>
                                <th scope="col" class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden xl:table-cell">Kondisi</th>';
    } elseif ($status === 'borrowed') {
        $html .= '<th scope="col" class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden lg:table-cell">Tanggal Pinjam</th>';
    } else {
        $html .= '<th scope="col" class="px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden lg:table-cell">Tanggal Dibuat</th>';
    }
    
    $html .= '
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">';
    
    foreach ($data as $item) {
        $statusBadge = '';
        switch($item['status']) {
            case 'pending_approval':
                $statusBadge = '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">Menunggu</span>';
                break;
            case 'borrowed':
                $statusBadge = '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">Dipinjam</span>';
                break;
            case 'returned':
                $statusBadge = '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-purple-100 text-purple-800">Dikembalikan</span>';
                break;
            case 'rejected':
                $statusBadge = '<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">Ditolak</span>';
                break;
        }
        
        $kondisiBadge = '-';
        if (!empty($item['kondisi_kembali'])) {
            switch($item['kondisi_kembali']) {
                case 'baik':
                    $kondisiBadge = '<span class="text-green-600 font-medium">Baik</span>';
                    break;
                case 'rusak':
                    $kondisiBadge = '<span class="text-yellow-600 font-medium">Rusak</span>';
                    break;
                case 'hilang':
                    $kondisiBadge = '<span class="text-red-600 font-medium">Hilang</span>';
                    break;
            }
        }
        
        $html .= '
            <tr class="hover:bg-gray-50">
                <td class="px-3 sm:px-6 py-4">
                    <div class="text-sm font-medium text-gray-900">' . htmlspecialchars($item['asset_name']) . '</div>
                    <div class="text-xs text-gray-500">' . htmlspecialchars($item['asset_number']) . '</div>
                    <div class="text-xs text-gray-400 md:hidden mt-1">
                        <span class="font-medium">workcenter:</span> ' . htmlspecialchars($item['workcenter'] ?? '-') . '
                    </div>
                </td>
                <td class="px-3 sm:px-6 py-4 text-sm text-gray-900 hidden md:table-cell">' . htmlspecialchars($item['workcenter'] ?? '-') . '</td>
                <td class="px-3 sm:px-6 py-4 text-sm text-gray-900 hidden lg:table-cell">' . htmlspecialchars($item['preference_pro'] ?? '-') . '</td>
                <td class="px-3 sm:px-6 py-4 whitespace-nowrap">' . $statusBadge . '</td>';
        
        if ($status === 'returned' || $status === 'all') {
            $tanggalKembali = !empty($item['tanggal_kembali']) ? date('d/m/y H:i', strtotime($item['tanggal_kembali'])) : '-';
            $html .= '
                <td class="px-3 sm:px-6 py-4 text-sm text-gray-500 hidden lg:table-cell whitespace-nowrap">' . $tanggalKembali . '</td>
                <td class="px-3 sm:px-6 py-4 text-sm hidden xl:table-cell">' . $kondisiBadge . '</td>';
        } elseif ($status === 'borrowed') {
            $tanggalPinjam = !empty($item['tanggal_peminjam']) ? date('d/m/y H:i', strtotime($item['tanggal_peminjam'])) : '-';
            $html .= '<td class="px-3 sm:px-6 py-4 text-sm text-gray-500 hidden lg:table-cell whitespace-nowrap">' . $tanggalPinjam . '</td>';
        } else {
            $tanggalDibuat = date('d/m/y H:i', strtotime($item['created_at']));
            $html .= '<td class="px-3 sm:px-6 py-4 text-sm text-gray-500 hidden lg:table-cell whitespace-nowrap">' . $tanggalDibuat . '</td>';
        }
        
        $html .= '</tr>';
    }
    
    $html .= '
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <div class="mt-4 text-sm text-gray-600">
            Total: <span class="font-semibold">' . count($data) . '</span> item
        </div>';
    
    // Add action buttons
    if ($status === 'borrowed') {
        $html .= '
            <div class="mt-6 p-4 bg-blue-50 rounded-lg border border-blue-200">
                <h4 class="font-medium text-blue-900 mb-3 flex items-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    Quick Actions
                </h4>
                <a href="return.php" class="inline-flex items-center justify-center w-full sm:w-auto px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition duration-200">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6"></path>
                    </svg>
                    Kembalikan Asset
                </a>
            </div>';
    }
    
    if ($status === 'pending_approval') {
        $html .= '
            <div class="mt-6 p-4 bg-yellow-50 rounded-lg border border-yellow-200">
                <h4 class="font-medium text-yellow-900 mb-2 flex items-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    Informasi
                </h4>
                <p class="text-sm text-yellow-800">Peminjaman Anda sedang menunggu persetujuan dari admin. Anda akan menerima notifikasi setelah disetujui atau ditolak.</p>
            </div>';
    }
    
    if ($status === 'returned') {
        $html .= '
            <div class="mt-6 p-4 bg-green-50 rounded-lg border border-green-200">
                <h4 class="font-medium text-green-900 mb-2 flex items-center">
                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    Riwayat Pengembalian
                </h4>
                <p class="text-sm text-green-800">Ini adalah daftar asset yang telah Anda kembalikan. Terima kasih atas tanggung jawab Anda dalam menjaga asset perusahaan.</p>
            </div>';
    }
    
    return $html;
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard Borrower - Peminjaman Tools Room</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'leaf-green': '#90EE90',
                        'dark-leaf': '#32CD32',
                        'soft-gray': '#F5F5F5',
                        'medium-gray': '#9CA3AF'
                    }
                }
            }
        }
    </script>
</head>
<body class="bg-soft-gray min-h-screen">
    <!-- Header -->
    <header class="bg-white shadow-sm border-b-2 border-leaf-green sticky top-0 z-50">
        <div class="max-w-7xl mx-auto px-3 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-3 sm:py-4">
                <div class="flex items-center space-x-2 sm:space-x-4">
                    <img src="../images/logo.png" alt="Logo" class="h-8 w-8 sm:h-10 sm:w-10" onerror="this.style.display='none'">
                    <div>
                        <h1 class="text-base sm:text-xl font-bold text-gray-800">Tools Room</h1>
                        <p class="text-xs sm:text-sm text-medium-gray">Dashboard Borrower</p>
                    </div>
                </div>
                <div class="flex items-center space-x-2 sm:space-x-4">
                    <div class="text-right hidden sm:block">
                        <span class="text-xs sm:text-sm text-gray-600">Halo, <?php echo $_SESSION['user_name']; ?></span>
                        <p class="text-xs text-medium-gray"><?php echo $borrower_org; ?></p>
                    </div>
                    <a href="../logout.php" class="bg-red-500 hover:bg-red-600 text-white px-3 py-1.5 sm:px-4 sm:py-2 rounded-lg text-xs sm:text-sm transition duration-200">
                        Logout
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="max-w-7xl mx-auto px-3 sm:px-6 lg:px-8 py-4 sm:py-8">
        <!-- Welcome Card -->
        <div class="bg-gradient-to-r from-leaf-green to-dark-leaf rounded-xl shadow-lg p-4 sm:p-6 mb-4 sm:mb-8 text-white">
            <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
                <div>
                    <h2 class="text-lg sm:text-2xl font-bold mb-1 sm:mb-2">Selamat Datang, <?php echo $_SESSION['user_name']; ?>!</h2>
                    <p class="text-sm sm:text-base opacity-90">Kelola peminjaman tools Anda dengan mudah</p>
                    <p class="text-xs sm:text-sm opacity-75 mt-1">Waktu: <?php echo date('d F Y, H:i'); ?> WIB</p>
                </div>
                <div class="text-left sm:text-right">
                    <div class="text-xl sm:text-2xl font-bold"><?php echo $available_count; ?></div>
                    <div class="text-xs sm:text-sm opacity-90">Asset Tersedia</div>
                </div>
            </div>
        </div>

        <!-- Stats Cards -->
        <div class="grid grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-6 mb-4 sm:mb-8">
            <!-- Pending Card -->
            <div class="bg-white rounded-xl shadow-lg p-3 sm:p-6 cursor-pointer hover:shadow-xl transition duration-300 border-l-4 border-yellow-400" onclick="showDetails('pending')">
                <div class="flex flex-col sm:flex-row items-start sm:items-center sm:justify-between gap-2">
                    <div class="w-full">
                        <p class="text-medium-gray text-xs sm:text-sm font-medium">Menunggu Persetujuan</p>
                        <p class="text-2xl sm:text-3xl font-bold text-yellow-600"><?php echo $pending_count; ?></p>
                    </div>
                    <div class="bg-yellow-100 p-2 sm:p-3 rounded-full self-end sm:self-auto">
                        <svg class="w-4 h-4 sm:w-6 sm:h-6 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <!-- Active Card -->
            <div class="bg-white rounded-xl shadow-lg p-3 sm:p-6 cursor-pointer hover:shadow-xl transition duration-300 border-l-4 border-green-400" onclick="showDetails('active')">
                <div class="flex flex-col sm:flex-row items-start sm:items-center sm:justify-between gap-2">
                    <div class="w-full">
                        <p class="text-medium-gray text-xs sm:text-sm font-medium">Sedang Dipinjam</p>
                        <p class="text-2xl sm:text-3xl font-bold text-green-600"><?php echo $active_count; ?></p>
                    </div>
                    <div class="bg-green-100 p-2 sm:p-3 rounded-full self-end sm:self-auto">
                        <svg class="w-4 h-4 sm:w-6 sm:h-6 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <!-- Returned Card -->
            <div class="bg-white rounded-xl shadow-lg p-3 sm:p-6 cursor-pointer hover:shadow-xl transition duration-300 border-l-4 border-purple-400" onclick="showDetails('returned')">
                <div class="flex flex-col sm:flex-row items-start sm:items-center sm:justify-between gap-2">
                    <div class="w-full">
                        <p class="text-medium-gray text-xs sm:text-sm font-medium">Dikembalikan</p>
                        <p class="text-2xl sm:text-3xl font-bold text-purple-600"><?php echo $returned_count; ?></p>
                    </div>
                    <div class="bg-purple-100 p-2 sm:p-3 rounded-full self-end sm:self-auto">
                        <svg class="w-4 h-4 sm:w-6 sm:h-6 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6"></path>
                        </svg>
                    </div>
                </div>
            </div>

            <!-- Total Card -->
            <div class="bg-white rounded-xl shadow-lg p-3 sm:p-6 cursor-pointer hover:shadow-xl transition duration-300 border-l-4 border-blue-400" onclick="showDetails('history')">
                <div class="flex flex-col sm:flex-row items-start sm:items-center sm:justify-between gap-2">
                    <div class="w-full">
                        <p class="text-medium-gray text-xs sm:text-sm font-medium">Total Riwayat</p>
                        <p class="text-2xl sm:text-3xl font-bold text-blue-600"><?php echo $total_count; ?></p>
                    </div>
                    <div class="bg-blue-100 p-2 sm:p-3 rounded-full self-end sm:self-auto">
                        <svg class="w-4 h-4 sm:w-6 sm:h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"></path>
                        </svg>
                    </div>
                </div>
            </div>
        </div>

        <!-- Organization Stats -->
        <div class="bg-white rounded-xl shadow-lg p-4 sm:p-6 mb-4 sm:mb-8">
            <h3 class="text-base sm:text-lg font-semibold text-gray-800 mb-3 sm:mb-4">Statistik Organisasi: <?php echo $borrower_org; ?></h3>
            <div class="grid grid-cols-2 gap-4 sm:gap-6">
                <div class="text-center p-3 sm:p-4 bg-soft-gray rounded-lg">
                    <div class="text-xl sm:text-2xl font-bold text-blue-600"><?php echo $org_stats['total_org_borrows']; ?></div>
                    <div class="text-xs sm:text-sm text-medium-gray mt-1">Total Peminjaman Organisasi</div>
                </div>
                <div class="text-center p-3 sm:p-4 bg-soft-gray rounded-lg">
                    <div class="text-xl sm:text-2xl font-bold text-green-600"><?php echo $org_stats['active_org_borrows']; ?></div>
                    <div class="text-xs sm:text-sm text-medium-gray mt-1">Sedang Dipinjam Organisasi</div>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-3 sm:gap-6 mb-4 sm:mb-8">
            <a href="borrow.php" class="bg-white rounded-xl shadow-lg p-4 sm:p-6 hover:shadow-xl transition duration-300 block">
                <div class="flex items-center space-x-3 sm:space-x-4">
                    <div class="bg-dark-leaf p-2 sm:p-3 rounded-full flex-shrink-0">
                        <svg class="w-5 h-5 sm:w-6 sm:h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                        </svg>
                    </div>
                    <div>
                        <h3 class="text-sm sm:text-lg font-semibold text-gray-800">Ajukan Peminjaman</h3>
                        <p class="text-xs sm:text-sm text-medium-gray">Scan atau input manual asset</p>
                    </div>
                </div>
            </a>

            <a href="return.php" class="bg-white rounded-xl shadow-lg p-4 sm:p-6 hover:shadow-xl transition duration-300 block">
                <div class="flex items-center space-x-3 sm:space-x-4">
                    <div class="bg-purple-500 p-2 sm:p-3 rounded-full flex-shrink-0">
                        <svg class="w-5 h-5 sm:w-6 sm:h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6"></path>
                        </svg>
                    </div>
                    <div>
                        <h3 class="text-sm sm:text-lg font-semibold text-gray-800">Pengembalian Asset</h3>
                        <p class="text-xs sm:text-sm text-medium-gray">Foto dan kembalikan asset</p>
                    </div>
                </div>
            </a>

            <a href="history.php" class="bg-white rounded-xl shadow-lg p-4 sm:p-6 hover:shadow-xl transition duration-300 block">
                <div class="flex items-center space-x-3 sm:space-x-4">
                    <div class="bg-blue-500 p-2 sm:p-3 rounded-full flex-shrink-0">
                        <svg class="w-5 h-5 sm:w-6 sm:h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                    </div>
                    <div>
                        <h3 class="text-sm sm:text-lg font-semibold text-gray-800">Riwayat Peminjaman</h3>
                        <p class="text-xs sm:text-sm text-medium-gray">Lihat semua riwayat lengkap</p>
                    </div>
                </div>
            </a>
        </div>

        <!-- Details Sections (Hidden by default) -->
        <div id="details-section" class="hidden mb-4 sm:mb-8">
            <div class="bg-white rounded-xl shadow-lg p-4 sm:p-6">
                <div class="flex justify-between items-center mb-3 sm:mb-4">
                    <h3 id="details-title" class="text-base sm:text-lg font-semibold text-gray-800"></h3>
                    <button onclick="closeDetails()" 
                            class="text-gray-400 hover:text-gray-600">
                        <svg class="w-5 h-5 sm:w-6 sm:h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
                <div id="details-content" class="text-sm sm:text-base"></div>
            </div>
        </div>

        <!-- Hidden data containers (Pre-rendered data from PHP) -->
        <div style="display:none;">
            <div id="data-pending"><?php echo renderDataTable($pending_data, 'pending_approval'); ?></div>
            <div id="data-borrowed"><?php echo renderDataTable($borrowed_data, 'borrowed'); ?></div>
            <div id="data-returned"><?php echo renderDataTable($returned_data, 'returned'); ?></div>
            <div id="data-all"><?php echo renderDataTable($all_data, 'all'); ?></div>
        </div>

        <!-- Recent Activity -->
        <?php if ($recent_history): ?>
        <div class="bg-white rounded-xl shadow-lg p-4 sm:p-6">
            <h3 class="text-base sm:text-lg font-semibold text-gray-800 mb-3 sm:mb-4">Aktivitas Terbaru</h3>
            <div class="space-y-3 sm:space-y-4">
                <?php foreach ($recent_history as $history): ?>
                <div class="flex flex-col sm:flex-row items-start sm:items-center justify-between p-3 sm:p-4 bg-soft-gray rounded-lg gap-2 sm:gap-0">
                    <div class="flex items-start sm:items-center space-x-2 sm:space-x-3 w-full sm:w-auto">
                        <div class="w-2 h-2 rounded-full flex-shrink-0 mt-1 sm:mt-0 <?php 
                            echo $history['status'] == 'pending_approval' ? 'bg-yellow-400' : 
                                ($history['status'] == 'borrowed' ? 'bg-green-400' : 
                                ($history['status'] == 'returned' ? 'bg-purple-400' : 'bg-gray-400')); 
                        ?>"></div>
                        <div class="min-w-0 flex-1">
                            <p class="font-medium text-gray-800 text-sm sm:text-base truncate"><?php echo htmlspecialchars($history['asset_name']); ?></p>
                            <p class="text-xs sm:text-sm text-medium-gray truncate"><?php echo htmlspecialchars($history['asset_number']); ?></p>
                            <?php if ($history['workcenter']): ?>
                            <p class="text-xs text-medium-gray truncate">workcenter: <?php echo htmlspecialchars($history['workcenter']); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="text-left sm:text-right w-full sm:w-auto pl-4 sm:pl-0">
                        <p class="text-xs sm:text-sm font-medium <?php 
                            echo $history['status'] == 'pending_approval' ? 'text-yellow-600' : 
                                ($history['status'] == 'borrowed' ? 'text-green-600' : 
                                ($history['status'] == 'returned' ? 'text-purple-600' : 'text-gray-600')); 
                        ?>">
                            <?php 
                            echo $history['status'] == 'pending_approval' ? 'Menunggu' : 
                                ($history['status'] == 'borrowed' ? 'Dipinjam' : 
                                ($history['status'] == 'returned' ? 'Dikembalikan' : 'Lainnya')); 
                            ?>
                        </p>
                        <p class="text-xs text-medium-gray">
                            <?php 
                            if ($history['status'] == 'returned' && $history['tanggal_kembali']) {
                                echo 'Kembali: ' . date('d/m/y H:i', strtotime($history['tanggal_kembali']));
                            } elseif ($history['tanggal_peminjam']) {
                                echo 'Pinjam: ' . date('d/m/y H:i', strtotime($history['tanggal_peminjam']));
                            } else {
                                echo 'Dibuat: ' . date('d/m/y H:i', strtotime($history['created_at']));
                            }
                            ?>
                        </p>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            
            <div class="mt-3 sm:mt-4 text-center">
                <a href="history.php" class="text-blue-600 hover:text-blue-800 text-xs sm:text-sm font-medium">
                    Lihat Semua Riwayat →
                </a>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script>
        function closeDetails() {
            document.getElementById('details-section').classList.add('hidden');
        }

        function showDetails(type) {
            const detailsSection = document.getElementById('details-section');
            const detailsTitle = document.getElementById('details-title');
            const detailsContent = document.getElementById('details-content');
            
            // Show details section
            detailsSection.classList.remove('hidden');
            
            // Set title and content based on type
            let dataId = '';
            switch(type) {
                case 'pending':
                    detailsTitle.textContent = 'Peminjaman Menunggu Persetujuan';
                    dataId = 'data-pending';
                    break;
                case 'active':
                    detailsTitle.textContent = 'Peminjaman Aktif';
                    dataId = 'data-borrowed';
                    break;
                case 'returned':
                    detailsTitle.textContent = 'Asset yang Dikembalikan';
                    dataId = 'data-returned';
                    break;
                case 'history':
                    detailsTitle.textContent = 'Semua Riwayat Peminjaman';
                    dataId = 'data-all';
                    break;
            }
            
            // Get pre-rendered data from hidden div
            const sourceData = document.getElementById(dataId);
            if (sourceData) {
                detailsContent.innerHTML = sourceData.innerHTML;
            } else {
                detailsContent.innerHTML = `
                    <div class="text-center py-8">
                        <p class="text-gray-600">Data tidak ditemukan</p>
                    </div>
                `;
            }
            
            // Scroll to details section
            setTimeout(() => {
                detailsSection.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
            }, 100);
        }

        // Show notification if there are pending items
        <?php if ($pending_count > 0): ?>
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Anda memiliki <?php echo $pending_count; ?> permintaan yang menunggu persetujuan');
        });
        <?php endif; ?>
    </script>
</body>
</html>